<?php

namespace Tests\Service;

use Doctrine\Common\Annotations\Reader;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\Persistence\Mapping\ClassMetadataFactory;
use Knp\Component\Pager\PaginatorInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Schema31\UtilityBundle\Service\IRicerca;
use Schema31\UtilityBundle\Service\Ricerca\Post;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Tests\Service\Fixture\RicercaModel;

class PostTest extends TestCase {
    /**
     * @var EntityManagerInterface|MockObject
     */
    private $entityManager;
    /**
     * @var PaginatorInterface|MockObject
     */
    private $paginator;
    /**
     * @var RequestStack|MockObject
     */
    private $requestStack;

    /**
     * @var SessionInterface|MockObject
     * @var mixed
     */
    private $session;
    /**
     * @var FormFactoryInterface|MockObject
     */
    private $formFactory;

    /**
     * @var FormInterface|MockObject
     */
    private $form;

    /**
     * @var Reader|MockObject
     */
    private $annotationReader;

    /**
     * @var ClassMetadataFactory|MockObject
     */
    private $metadataFactory;

    private $result = [];

    /**
     * @var EntityRepository|MockObject
     */
    private $repository;

    public function getResult() {
        return $this->result;
    }

    public function setUp() {
        $this->paginator = $this->createMock(PaginatorInterface::class);
        $request = new Request([]);
        $this->requestStack = $this->createMock(RequestStack::class);
        $this->requestStack->method('getCurrentRequest')->willReturn($request);
        $this->session = $this->createMock(SessionInterface::class);

        $this->formFactory = $this->createMock(FormFactoryInterface::class);
        $this->form = $this->createMock(FormInterface::class);
        $this->formFactory->method('create')->willReturn($this->form);

        $this->entityManager = $this->createMock(EntityManagerInterface::class);
        $this->repository = $this->createMock(EntityRepository::class);

        $this->entityManager->method('getRepository')->willreturn($this->repository);

        $this->annotationReader = $this->createMock(Reader::class);

        $this->metadataFactory = $this->createMock(ClassMetadataFactory::class);
        $this->entityManager->method('getMetadataFactory')->willReturn($this->metadataFactory);
    }

    public function testPulisci(): void {
        $ricerca = $this->createService();
        $res = $ricerca->pulisci([
            'form_type' => 'form_type',
            'repository' => 'repository',
            'repository_method' => 'repository_method',
            'data' => new RicercaModel(),
        ]);

        $this->assertNull($res);
    }

    private function createService(int $defaultNumberElemens = 5): IRicerca {
        return new Post(
            $defaultNumberElemens,
            $this->entityManager,
            $this->session,
            $this->paginator,
            $this->requestStack,
            $this->formFactory,
            $this->annotationReader
        );
    }

    public function testRicercaSenzaSubmit(): void {
        $ricerca = $this->createService();
        $res = $ricerca->ricerca([
            'form_type' => 'form_type',
            'repository' => 'repository',
            'repository_method' => 'getResult',
            'data' => new RicercaModel(),
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertArrayHasKey('form_ricerca', $res);
        $this->assertArrayHasKey('filtro_attivo', $res);
        $this->assertArrayHasKey('form', $res);

        $this->assertFalse($res['filtro_attivo']);
    }

    public function testSenzaRicercaConLimiteVisibile(): void {
        $ricerca = $this->createService();
        $res = $ricerca->ricerca([
            'form_type' => 'form_type',
            'repository' => 'repository',
            'repository_method' => 'getResult',
            'data' => new RicercaModel(),
            'show_elements_number' => true,
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertArrayHasKey('form_ricerca', $res);
        $this->assertArrayHasKey('filtro_attivo', $res);
        $this->assertArrayHasKey('form', $res);

        $this->assertFalse($res['filtro_attivo']);
    }

    public function testRicercaFormSubmitted(): void {
        $this->form->method('isSubmitted')->willReturn(true);

        $ricerca = $this->createService();
        $res = $ricerca->ricerca([
            'form_type' => 'form_type',
            'repository' => 'repository',
            'repository_method' => 'getResult',
            'data' => new RicercaModel(),
        ]);
        $this->assertArrayHasKey('risultato', $res);
        $this->assertArrayHasKey('form_ricerca', $res);
        $this->assertArrayHasKey('filtro_attivo', $res);
        $this->assertArrayHasKey('form', $res);

        $this->assertFalse($res['filtro_attivo']);
    }

    public function testRicercaFormSubmittedConSessioneNoEntity(): void {
        $modelloRicercaPrototype = new RicercaModel();
        $modelloRicerca = new RicercaModel();
        $modelloRicerca->collection = new ArrayCollection([1, '----', new RicercaModel()]);
        $this->metadataFactory->method('isTransient')->willReturn(true);
        $this->form->method('isSubmitted')->willReturn(true);
        $this->session->method('has')->with(RicercaModel::class)->willReturn(true);
        $this->session->method('get')->will($this->returnValueMap([
            [RicercaModel::class, $modelloRicercaPrototype, $modelloRicerca, ],
            ["Tests\Service\Fixture\RicercaModel_elements", null, 5],
        ]));
        $ricerca = $this->createService();
        $res = $ricerca->ricerca([
            'form_type' => 'form_type',
            'repository' => 'repository',
            'repository_method' => 'getResult',
            'data' => $modelloRicercaPrototype,
        ]);
        $this->assertArrayHasKey('risultato', $res);
        $this->assertArrayHasKey('form_ricerca', $res);
        $this->assertArrayHasKey('filtro_attivo', $res);
        $this->assertArrayHasKey('form', $res);

        $this->assertTrue($res['filtro_attivo']);
    }

    public function testRicercaFormSubmittedConSessioneEntity(): void {
        $modelloRicercaPrototype = new RicercaModel();
        $modelloRicerca = new RicercaModel();
        $modelloRicerca->ricerca = new RicercaModel();
        $this->metadataFactory->method('isTransient')->willReturn(false);
        $this->form->method('isSubmitted')->willReturn(true);
        $this->session->method('has')->with(RicercaModel::class)->willReturn(true);
        $this->session->method('get')->will($this->returnValueMap([
            [RicercaModel::class, $modelloRicercaPrototype, $modelloRicerca, ],
            ["Tests\Service\Fixture\RicercaModel_elements", null, 5],
        ]));
        $classMetadata = $this->createMock(ClassMetadata::class);
        $this->entityManager->method('getClassMetadata')->willReturn($classMetadata);
        $classMetadata->method('getIdentifierValues')->willReturn([]);
        $classMetadata->rootEntityName = 'nome_classe';
        $this->repository->expects($this->once())->method('findOneBy');
        $ricerca = $this->createService();
        $res = $ricerca->ricerca([
            'form_type' => 'form_type',
            'repository' => 'repository',
            'repository_method' => 'getResult',
            'data' => $modelloRicercaPrototype,
        ]);
        $this->assertArrayHasKey('risultato', $res);
        $this->assertArrayHasKey('form_ricerca', $res);
        $this->assertArrayHasKey('filtro_attivo', $res);
        $this->assertArrayHasKey('form', $res);

        $this->assertTrue($res['filtro_attivo']);
    }
}
