<?php

namespace Tests\Repository;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Query\Expr;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Schema31\UtilityBundle\Repository\LazyQueryBuilder;
use Tests\Repository\Fixture\Enumerazione;

class LazyQueryBuilderTest extends TestCase {
    /**
     * @var EntityManagerInterface|MockObject
     */
    private $em;

    /**
     * @var LazyQueryBuilder
     */
    private $lazyQb;

    public function setUp() {
        parent::setUp();
        $this->em = $this->createMock(EntityManagerInterface::class);
        $this->em->method('getExpressionBuilder')->willReturn(new Expr());
        $this->lazyQb = new LazyQueryBuilder($this->em);
        $this->lazyQb->select('x')
        ->from('entity', 'x');
    }

    public function testLazyLeftJoin(): void {
        $this->lazyQb->lazyLeftJoin('x.b', 'b');
        $this->lazyQb->use('b');

        /** @var Expr\Join $join */
        $join = $this->getPart('join', 'x')[0];

        $this->assertEquals('LEFT', $join->getJoinType());
        $this->assertEquals('x.b', $join->getJoin());
        $this->assertEquals('b', $join->getAlias());
    }

    public function testMultipleLeftJoin(): void {
        $this->lazyQb->lazyLeftJoin('x.b', 'b');
        $this->lazyQb->lazyLeftJoin('b.a', 'a');

        $this->lazyQb->use('a');

        /** @var Expr\Join $join */
        $join = $this->getPart('join', 'x')[0];

        $this->assertEquals('LEFT', $join->getJoinType());
        $this->assertEquals('x.b', $join->getJoin());
        $this->assertEquals('b', $join->getAlias());
    }

    private function getPart(string $dqlPart, ?string $alias = null) {
        $joinParts = $this->lazyQb->getDQLPart($dqlPart);
        if (\is_null($alias)) {
            return $joinParts;
        }
        return $joinParts[$alias];
    }

    public function testAddOrderByString(): void {
        $this->lazyQb->addOrderBy('x', 'ASC');

        /** @var Expr\OrderBy */
        $sort = $this->getPart('orderBy')[0];
        $parts = $sort->getParts();

        $this->assertCount(1, $parts);
        $this->assertEquals('x ASC', $parts[0]);
    }

    public function testAddOrderByObject(): void {
        $order = new Expr\OrderBy('x', 'DESC');
        $this->lazyQb->addOrderBy($order);
        /** @var Expr\OrderBy */
        $sort = $this->getPart('orderBy')[0];
        $parts = $sort->getParts();

        $this->assertCount(1, $parts);
        $this->assertEquals('x DESC', $parts[0]);
    }

    public function testAddEqualFilter(): void {
        $this->lazyQb->addEqualFilter('x', 1);

        $parts = $this->getWhereParts();

        self::assertStringStartsWith('x = :x', $parts[0]);
    }

    private function getWhereParts(): array {
        /** @var Expr\AndX */
        $where = $this->getPart('where');
        $parts = \is_null($where) ? [] : $where->getParts();

        return $parts;
    }

    public function testAddLikeFilter(): void {
        $this->lazyQb->addLikeFilter('x', 'test');

        $parts = $this->getWhereParts();

        self::assertStringStartsWith('x LIKE :x', $parts[0]);
    }

    public function testAddEnumFilter(): void {
        $e = Enumerazione::VIEW();
        $this->lazyQb->addEnumFilter('x', $e);

        $parts = $this->getWhereParts();

        self::assertCount(1, $parts);
        self::assertStringStartsWith('x = :x', $parts[0]);
    }

    public function testAddEnumFilterNull(): void {
        $this->lazyQb->addEnumFilter('x', null);

        $parts = $this->getWhereParts();

        $this->assertEmpty($parts);
    }

    public function testAddSingleDateFilter(): void {
        $this->lazyQb->addSingleDateFilter('x', new \DateTime());

        $parts = $this->getWhereParts();

        self::assertStringStartsWith('x BETWEEN :x', $parts[0]);
    }

    public function testNullAddBetweenFilter(): void {
        $this->lazyQb->addBetweenFilter('x.data', null, null);
        $parts = $this->getPart('where');

        $this->assertNull($parts);
    }
}
