<?php

namespace Schema31\UtilityBundle\Repository;

use Doctrine\ORM\Query\Expr;
use Doctrine\ORM\QueryBuilder;
use MyCLabs\Enum\Enum;

/**
 * Description of LazyQueryBuilder
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class LazyQueryBuilder extends QueryBuilder {
    /** @var array */
    private $relations = [];

    public function lazyLeftJoin($join, $alias): self {
        $this->relations[$alias] = ['join' => $join, 'used' => false];

        return $this;
    }

    public function use(string $alias): self {
        if (isset($this->relations[$alias]) && !$this->relations[$alias]['used']) {
            $this->relations[$alias]['used'] = true;
            $parentAlias = \explode('.', $this->relations[$alias]['join'])[0];

            $this->use($parentAlias);

            $this->leftJoin($this->relations[$alias]['join'], $alias);
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function addOrderBy($sort, $order = null): self {
        if ($sort instanceof Expr\OrderBy) {
            $parts = $sort->getParts()[0];
            $tokens = \explode(' ', $parts);
            $sort = $tokens[0];
            $order = $order ?? $tokens[1] ?? null;
        }
        $this->use($sort);

        return parent::addOrderBy($sort, $order);
    }

    public function addComparisonFilter(string $relazione, $data, string $operator): self {
        if (!\is_null($data)) {
            $this->use($relazione);
            $normalizedField = $this->normalizeParam($relazione) . \md5($operator);
            $this->andWhere("$relazione $operator :$normalizedField");
            // $normalizedData = \is_object($data) && (\get_class($data) == \DateTime::class) ? $data->format('Y-m-d H:i:s') : $data;
            $this->setParameter($normalizedField, $data);
        }

        return $this;
    }

    private function normalizeParam(string $relation): string {
        return \str_replace('.', '_', $relation);
    }

    public function addEnumFilter(string $filter, ?Enum $data): self {
        if(\is_null($data)) {
            return $this;
        }
        $normalizedData = $data->getKey();

        return $this->addComparisonFilter($filter, $normalizedData, '=');
    }

    public function addEqualFilter(string $filter, $data): self {
        return $this->addComparisonFilter($filter, $data, '=');
    }

    public function addLikeFilter(string $filter, $data): self {
        $normalizedData = "%$data%";

        return $this->addComparisonFilter($filter, $normalizedData, 'LIKE');
    }

    public function addSingleDateFilter(string $filter, ?\DateTime $data): self {
        if (!\is_null($data)) {
            $inizio = $data->setTime(0, 0, 0);
            $fine = (clone $inizio)->modify("+1 day");
            $this->addBetweenFilter($filter, $inizio, $fine);
        }

        return $this;
    }

    public function addBetweenFilter(string $filter, ?\DateTime $start, ?\DateTime $end): self {
        if (\is_null($start) && \is_null($end)) {
            return $this;
        }
        $this->use($filter);
        $start = $start ?? new \DateTime('0000-01-01');
        $end = $end ?? new \DateTime('9999-12-31');

        $normalizedField = $this->normalizeParam($filter);
        $this->andWhere($this->expr()->between($filter, ":$normalizedField" . "_1", ":$normalizedField" . "_2"));
        $this->setParameter($normalizedField . "_1", $start);
        $this->setParameter($normalizedField . "_2", $end);

        return $this;
    }
}
