<?php

namespace Tests\Service;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Knp\Component\Pager\PaginatorInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Schema31\UtilityBundle\Service\IRicerca;
use Schema31\UtilityBundle\Service\Ricerca\Get;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\OptionsResolver\Exception\MissingOptionsException;

class GetTest extends TestCase {
    /**
     * @var EntityManagerInterface|MockObject
     */
    private $entityManager;
    /**
     * @var PaginatorInterface|MockObject
     */
    private $paginator;
    /**
     * @var RequestStack|MockObject
     */
    private $requestStack;
    /**
     * @var FormFactoryInterface|MockObject
     */
    private $formFactor;
    /**
     * @var FormBuilderInterface|MockObject
     */
    private $formBuilder;
    /**
     * @var FormInterface|MockObject
     */
    private $form;

    private $result = [];

    public function getResult() {
        return $this->result;
    }

    public function setUp() {
        $this->paginator = $this->createMock(PaginatorInterface::class);
        $request = new Request([]);
        $this->requestStack = $this->createMock(RequestStack::class);
        $this->requestStack->method('getCurrentRequest')->willReturn($request);

        $this->formFactor = $this->createMock(FormFactoryInterface::class);
        $this->formBuilder = $this->createMock(FormBuilderInterface::class);
        $this->formFactor->method('createBuilder')->willReturn($this->formBuilder);
        $this->form = $this->createMock(FormInterface::class);
        $this->formBuilder->method('setMethod')->willReturn($this->formBuilder);
        $this->formBuilder->method('getForm')->willReturn($this->form);

        $this->entityManager = $this->createMock(EntityManagerInterface::class);
        $this->entityManager->method('getRepository')->willreturn($this);
    }

    public function testPulisci(): void {
        $ricerca = $this->createService();
        $res = $ricerca->pulisci([]);

        $this->assertNull($res);
    }

    private function createService(int $defaultNumberElemens = 5): IRicerca {
        return new Get(
            $defaultNumberElemens,
            $this->entityManager,
            $this->paginator,
            $this->requestStack,
            $this->formFactor
        );
    }

    public function testRepositoryNonIndicato(): void {
        $ricerca = $this->createService();
        $this->expectException(MissingOptionsException::class);
        $ricerca->ricerca([
            'form_type' => '',
            'repository_method' => 'repository_method',
        ]);
    }

    public function testRepositoryMethodNonIndicato(): void {
        $ricerca = $this->createService();
        $this->expectException(MissingOptionsException::class);
        $ricerca->ricerca([
            'form_type' => '',
            'repository' => 'repository',
        ]);
    }

    public function testRepositoryFormTypeNonIndicato(): void {
        $ricerca = $this->createService();
        $this->expectException(MissingOptionsException::class);
        $ricerca->ricerca([
            'repository_method' => 'ricerca',
            'repository' => 'repository',
        ]);
    }

    public function testRicercaNonSottomesso(): void {
        $this->paginator->expects($this->once())->method('paginate')->willReturn(['risultato']);
        $ricerca = $this->createService();
        $res = $ricerca->ricerca([
            'repository_method' => 'getResult',
            'form_type' => 'form_type',
            'repository' => 'repository',
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertFalse($res['filtro_attivo']);
    }

    public function testRicercaSottomesso(): void {
        $this->paginator->expects($this->once())->method('paginate')->willReturn(['risultato']);
        $ricerca = $this->createService();
        $this->form->method('isSubmitted')->willReturn(true);

        $res = $ricerca->ricerca([
            'repository_method' => 'getResult',
            'form_type' => 'form_type',
            'repository' => 'repository',
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertTrue($res['filtro_attivo']);
    }

    public function testRicercaConMostraElementi(): void {
        $this->paginator->expects($this->once())->method('paginate')->willReturn(['risultato']);
        $ricerca = $this->createService();
        $this->form->method('has')->willReturn(true);
        $formNumero = $this->createMock(FormInterface::class);
        $this->form->method('get')->willReturn($formNumero);
        $this->form->method('isSubmitted')->willReturn(true);

        $res = $ricerca->ricerca([
            'repository_method' => 'getResult',
            'form_type' => 'form_type',
            'repository' => 'repository',
            'show_elements_number' => true,
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertTrue($res['filtro_attivo']);
    }

    public function testRicercaConNumeroInfinito(): void {
        $this->paginator->expects($this->once())->method('paginate')->willReturn(['risultato']);
        $ricerca = $this->createService();
        $this->form->method('isSubmitted')->willReturn(true);

        $res = $ricerca->ricerca([
            'repository_method' => 'getResult',
            'form_type' => 'form_type',
            'repository' => 'repository',
            'show_elements_number' => true,
            'elements_per_page' => function () {
                return \INF;
            },
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertTrue($res['filtro_attivo']);
    }

    public function testRicercaConQueryBuilder(): void {
        $this->paginator->expects($this->once())->method('paginate')->willReturn(['risultato']);
        $ricerca = $this->createService();
        $this->form->method('isSubmitted')->willReturn(true);
        $this->result = $this->createMock(QueryBuilder::class);
        $q = $this->createMock(AbstractQuery::class);
        $this->result->method('getQuery')->willReturn($q);

        $res = $ricerca->ricerca([
            'repository_method' => 'getResult',
            'form_type' => 'form_type',
            'repository' => 'repository',
            'show_elements_number' => true,
            'elements_per_page' => function () {
                return \INF;
            },
        ]);

        $this->assertArrayHasKey('risultato', $res);
        $this->assertTrue($res['filtro_attivo']);
    }
}
