<?php

namespace Schema31\UtilityBundle\Service\Ricerca;

use Doctrine\Common\Annotations\Reader;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

/**
 * Servizio che cerca di standardizare la ricerca nel sistema
 * Il servizio ha un metodo ricerca che genera la form di ricerca,effettua la query e ritorna i risultati
 */
class Post extends AbstractMethod {
    /**
     * @var SessionInterface
     */
    protected $session;

    /**
     * @var int
     */
    protected $numeroMassimoElementi;

    /**
     * @var object|array|null
     */
    protected $prototype;

    /**
     * @var Reader
     */
    protected $annotationReader;

    public function __construct(
            $numeroElementi,
            EntityManagerInterface $entityManager,
            SessionInterface $session,
            PaginatorInterface $paginator,
            RequestStack $requestStack,
            FormFactoryInterface $formFactory,
            Reader $annotationReader
        ) {
        $this->entityManager = $entityManager;
        $this->session = $session;
        $this->paginator = $paginator;
        $this->request = $requestStack->getCurrentRequest();
        $this->formFactory = $formFactory;
        $this->numeroDefaultElementi = $numeroElementi;
        $this->annotationReader = $annotationReader;
    }

    public function pulisci(array $options): void {
        $this->resolveOptions($options);
        $nomeOggettoInSessione = $this->getNomeOggettoInSessione();
        $this->session->remove($nomeOggettoInSessione);
        $this->session->remove($nomeOggettoInSessione . '_elements');
    }

    public function ricerca(array $options = []): array {
        $parameters = $this->resolveOptions($options);
        $nomeOggettoInSession = $this->getNomeOggettoInSessione();

        $this->data = $this->session->get($nomeOggettoInSession, $this->prototype);
        if ($this->session->has($nomeOggettoInSession)) {
            $this->rendeManaged();
        }

        $this->limit = $this->getNumeroElementiPerPagina($parameters['elements_per_page']);

        $formRicerca = $this->formFactory->create($options['form_type'], $this->data, $options['form_options'] ?? []);

        $this->addChoiceElementsPerPage($formRicerca, $parameters);

        $nomeParametroPagina = $parameters['page_index'];
        $numeroPaginaRicerca = $this->request->get($nomeParametroPagina, 1);
        $formRicerca->handleRequest($this->request);
        $this->aggiornaElementiDaVisualizzare($formRicerca, $parameters['elements_per_page']);
        if ($formRicerca->isSubmitted()) {
            $this->data = $formRicerca->getData();
            $this->session->set($nomeOggettoInSession, $this->data);
            $numeroPaginaRicerca = 1;
            $this->session->set($nomeOggettoInSession . '_elements', $this->limit);
        }

        $filtroAttivo = true == $this->session->has($nomeOggettoInSession);

        $pagination_object = [];
        if ($filtroAttivo || true === $parameters['empty_search']) {
            $repository = $this->entityManager->getRepository($parameters['repository']);
            $pagination_object = \call_user_func([$repository, $parameters['repository_method']], $this->data);
        }
        $pagination = $this->paginate($pagination_object, $numeroPaginaRicerca);

        return [
            "risultato" => $pagination,
            "form_ricerca" => $formRicerca->createView(),
            "filtro_attivo" => $filtroAttivo,
            'form' => $formRicerca,
        ];
    }

    private function resolveOptions(array $options): array {
        $resolver = new OptionsResolver();
        $resolver->setDefaults([
            'data' => null,
            'choice_elements_per_page' => [
                '5' => 5,
                '10' => 10,
                '25' => 25,
                '50' => 50,
                '75' => 75,
                '100' => 100,
            ],
            'elements_per_page' => null,
            'page_index' => 'page',
            'show_elements_number' => false,
            'empty_search' => true,
        ]);
        $resolver->setRequired('form_type');
        $resolver->setRequired('repository');
        $resolver->setRequired('repository_method');

        $resolver->setAllowedTypes('form_type', 'string');
        $resolver->setAllowedTypes('repository', 'string');
        $resolver->setAllowedTypes('repository_method', 'string');
        $resolver->setAllowedTypes('page_index', 'string');
        $resolver->setAllowedTypes('choice_elements_per_page', 'int[]');
        $resolver->setAllowedTypes('elements_per_page', ['int', 'callable', 'null']);
        $resolver->setAllowedTypes('show_elements_number', 'bool');
        $resolver->setAllowedTypes('empty_search', 'bool');

        $parameters = $resolver->resolve($options);

        $this->prototype = $parameters['data'];

        return $parameters;
    }

    /** nel caso si ripeschi un oggetto dalla sessione le entity sono tutte detached
     * quindi nel caso si ricerchi usando oggetti di tipo Entity cerco di farne il merge
     * in caso di errore del tipo "Entities passed to the choice field must be managed"
     * controlla che la base class non abbia attributi private
     */
    protected function rendeManaged(): void {
        $modelloReflection = new \ReflectionClass($this->data);
        $currentProps = $modelloReflection->getProperties();
        $parent = $modelloReflection->getParentClass();
        $parentProps = $parent ? $parent->getProperties() : [];
        $props = \array_merge($currentProps, $parentProps);

        foreach ($props as $prop) {
            $prop->setAccessible(true);
            $valore = $prop->getValue($this->data);

            if (!\is_object($valore) || $valore instanceof \DateTime) {
                continue;
            }

            $new_valore = $valore instanceof Collection ? $valore->map(function ($val) {
                return $this->getValore($val);
            }) :
                $this->getValore($valore);

            $prop->setValue($this->data, $new_valore);
        }
    }

    /**
     * @param object $valore
     * @return object|null
     */
    protected function getValore($valore) {
        if (false === \is_object($valore)) {
            return $valore;
        }
        $proxy_class_name = \get_class($valore);
        if (!$this->isEntity($proxy_class_name)) {
            return $valore;
        }
        $classMetadata = $this->entityManager->getClassMetadata($proxy_class_name);
        $class_name = $classMetadata->rootEntityName;
        $repository = $this->entityManager->getRepository($class_name);
        $key = $classMetadata->getIdentifierValues($valore);
        $new_valore = $repository->findOneBy($key);

        return $new_valore;
    }

    private function isEntity(string $class): bool {
        return !$this->entityManager->getMetadataFactory()->isTransient($class);
    }

    /**
     * @return float|int
     * @param mixed $opzione
     */
    private function getNumeroElementiPerPagina($opzione) {
        $opzioneNormalizzata = \is_callable($opzione) ? null : $opzione;
        $nomeOggettoInSession = $this->getNomeOggettoInSessione() . '_elements';
        $valoreInSessione = $this->session->get($nomeOggettoInSession, null);
        $numElementi = $this->limit ??
                $valoreInSessione ??
                $opzioneNormalizzata ??
                $this->numeroDefaultElementi ??
                0;

        return $numElementi;
    }

    private function getNomeOggettoInSessione(): string {
        $modelloReflection = new \ReflectionClass($this->prototype);
        return $modelloReflection->getName();
    }
}
