<?php

namespace Schema31\UtilityBundle\Service\Ricerca;

use Doctrine\Common\Annotations\Reader;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Persistence\Proxy;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Knp\Component\Pager\PaginatorInterface;
use Schema31\UtilityBundle\Service\IRicerca;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

/**
 * Servizio che cerca di standardizare la ricerca nel sistema
 * Il servizio ha un metodo ricerca che genera la form di ricerca,effettua la query e ritorna i risultati
 */
class Post implements IRicerca {
    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;
    /**
     * @var SessionInterface
     */
    protected $session;

    /**
     * @var PaginatorInterface
     */
    protected $paginator;
    /**
     * @var Request
     */
    protected $request;
    /**
     * @var FormFactoryInterface
     */
    protected $formFactory;

    /**
     * @var object|array|null
     */
    protected $data;

    /**
     * @var int
     */
    protected $numeroMassimoElementi;
    /**
     * @var int
     */
    protected $numeroDefaultElementi;

    /**
     * @var object|array|null
     */
    protected $prototype;

    /**
     * @var Reader
     */
    protected $annotationReader;

    /**
     * @var int
     */
    protected $limit;

    public function __construct(
            $numeroElementi,
            EntityManagerInterface $entityManager,
            SessionInterface $session,
            PaginatorInterface $paginator,
            RequestStack $requestStack,
            FormFactoryInterface $formFactory,
            Reader $annotationReader
        ) {
        $this->entityManager = $entityManager;
        $this->session = $session;
        $this->paginator = $paginator;
        $this->request = $requestStack->getCurrentRequest();
        $this->formFactory = $formFactory;
        $this->numeroDefaultElementi = $numeroElementi;
        $this->annotationReader = $annotationReader;
    }

    public function pulisci(array $options): void {
        $this->resolveOptions($options);
        $nomeOggettoInSessione = $this->getNomeOggettoInSessione();
        $this->session->remove($nomeOggettoInSessione);
        $this->session->remove($nomeOggettoInSessione . '_elements');
    }

    public function ricerca(array $options = []): array {
        $parameters = $this->resolveOptions($options);
        $nomeOggettoInSession = $this->getNomeOggettoInSessione();

        $this->data = $this->session->get($nomeOggettoInSession, $this->prototype);
        if ($this->session->has($nomeOggettoInSession)) {
            $this->rendeManaged();
        }

        $this->limit = $this->getNumeroElementiPerPagina($parameters['elements_per_page']);

        $formRicerca = $this->formFactory->create($options['form_type'], $this->data);

        if ($parameters['show_elements_number']) {
            $listaElementiPerPagina = $parameters['choice_elements_per_page'];
            $listaElementiPerPagina["{$this->limit}"] = $this->limit;
            $numeroElementiDefault = \is_callable($parameters['elements_per_page']) ? $parameters['elements_per_page']($formRicerca) :
                \is_null($parameters['elements_per_page']) ? \reset($listaElementiPerPagina) :
                $parameters['elements_per_page'];
            if (\is_int($numeroElementiDefault)) {
                $listaElementiPerPagina["$numeroElementiDefault"] = $numeroElementiDefault;
            }
            \asort($listaElementiPerPagina);

            $formRicerca->add('numero_elementi', ChoiceType::class, [
                'required' => false,
                'choices' => $listaElementiPerPagina,
                'placeholder' => false,
                'label' => 'Elementi per pagina',
                'mapped' => false,
                'data' => $numeroElementiDefault,
            ]);
        }

        $nomeParametroPagina = $parameters['page_index'];
        $numeroPaginaRicerca = $this->request->get($nomeParametroPagina, 1);
        $formRicerca->handleRequest($this->request);
        if ($formRicerca->isSubmitted()) {
            $this->data = $formRicerca->getData();
            $this->session->set($nomeOggettoInSession, $this->data);
            $numeroPaginaRicerca = 1;
            $this->limit = $this->getElementiDaVisualizzare($formRicerca, $parameters['elements_per_page']);
            $this->session->set($nomeOggettoInSession . '_elements', $this->limit);
        }

        $filtroAttivo = $this->session->has($nomeOggettoInSession);

        $pagination_object = [];
        if ($filtroAttivo || true === $parameters['empty_search']) {
            $repository = $this->entityManager->getRepository($parameters['repository']);
            $pagination_object = \call_user_func([$repository, $parameters['repository_method']], $this->data);
        }
        $pagination = ['items' => []];
        if (\is_int($this->limit)) {
            $pagination = $this->paginator->paginate(
                $pagination_object,
                $numeroPaginaRicerca,
                $this->limit
            );
        } else {
            $result = $pagination_object;
            if ($pagination_object instanceof QueryBuilder) {
                $pagination_object = $pagination_object->getQuery();
            }
            if ($pagination_object instanceof Query) {
                $result = $pagination_object->getResult();
            }
            $pagination = [
                'items' => $result,
                'getTotalItemCount' => \count($result),
            ];
        }

        return [
            "risultato" => $pagination,
            "form_ricerca" => $formRicerca->createView(),
            "filtro_attivo" => $filtroAttivo,
            'form' => $formRicerca,
        ];
    }

    private function resolveOptions(array $options): array {
        $resolver = new OptionsResolver();
        $resolver->setDefaults([
            'data' => null,
            'choice_elements_per_page' => [
                '5' => 5,
                '10' => 10,
                '25' => 25,
                '50' => 50,
                '75' => 75,
                '100' => 100,
            ],
            'elements_per_page' => null,
            'page_index' => 'page',
            'show_elements_number' => false,
            'empty_search' => true,
        ]);
        $resolver->setRequired('form_type');
        $resolver->setRequired('repository');
        $resolver->setRequired('repository_method');

        $resolver->setAllowedTypes('form_type', 'string');
        $resolver->setAllowedTypes('repository', 'string');
        $resolver->setAllowedTypes('repository_method', 'string');
        $resolver->setAllowedTypes('page_index', 'string');
        $resolver->setAllowedTypes('choice_elements_per_page', 'int[]');
        $resolver->setAllowedTypes('elements_per_page', ['int', 'callable', 'null']);
        $resolver->setAllowedTypes('show_elements_number', 'bool');
        $resolver->setAllowedTypes('empty_search', 'bool');

        $parameters = $resolver->resolve($options);

        $this->prototype = $parameters['data'];

        return $parameters;
    }

    /** nel caso si ripeschi un oggetto dalla sessione le entity sono tutte detached
     * quindi nel caso si ricerchi usando oggetti di tipo Entity cerco di farne il merge
     * in caso di errore del tipo "Entities passed to the choice field must be managed"
     * controlla che la base class non abbia attributi private
     */
    protected function rendeManaged(): void {
        $modelloReflection = new \ReflectionClass($this->data);
        $currentProps = $modelloReflection->getProperties();
        $parent = $modelloReflection->getParentClass();
        $parentProps = $parent ? $parent->getProperties() : [];
        $props = \array_merge($currentProps, $parentProps);

        foreach ($props as $prop) {
            $prop->setAccessible(true);
            $valore = $prop->getValue($this->data);

            if (!\is_object($valore) || $valore instanceof \DateTime) {
                continue;
            }

            $new_valore = $this->getValore($valore);
            if ($valore instanceof Collection) {
                $new_valore = new ArrayCollection();
                foreach ($valore as $val) {
                    $new_valore_elemento = $this->getValore($val);
                    $new_valore->add($new_valore_elemento);
                }
            }
            $prop->setValue($this->data, $new_valore);
        }
    }

    /**
     * @param object $valore
     * @return object|null
     */
    protected function getValore($valore) {
        $proxy_class_name = \get_class($valore);
        if (!$this->isEntity($valore)) {
            return $valore;
        }
        $classMetadata = $this->entityManager->getClassMetadata($proxy_class_name);
        $class_name = $classMetadata->rootEntityName;
        $key = $classMetadata->getIdentifierValues($valore);
        $new_valore = $this->entityManager->getRepository($class_name)->findOneBy($key);

        return $new_valore;
    }

    /**
     * @param mixed $object
     */
    private function isEntity($object): bool {
        if (is_object($object)) {
            $object = ($object instanceof Proxy)
            ? get_parent_class($object)
            : get_class($object);
        }

        return !$this->entityManager->getMetadataFactory()->isTransient($object);
    }

    /**
     * @return float|int
     */
    private function getNumeroElementiPerPagina($opzione) {
        $opzioneNormalizzata = \is_callable($opzione) ? null : $opzione;
        $nomeOggettoInSession = $this->getNomeOggettoInSessione() . '_elements';
        $valoreInSessione = $this->session->get($nomeOggettoInSession, null);
        $numElementi = $this->limit ??
                $valoreInSessione ??
                $opzioneNormalizzata ??
                $this->numeroDefaultElementi ??
                0;

        return $numElementi;
    }

    private function getNomeOggettoInSessione(): string {
        $modelloReflection = new \ReflectionClass($this->prototype);
        return $modelloReflection->getName();
    }

    private function getElementiDaVisualizzare(FormInterface $form, $parametro) {
        if (\is_null($parametro)) {
            $this->limit = $this->numeroDefaultElementi;
        }
        if (\is_callable($parametro)) {
            $this->limit = $parametro($form);
        }
        if ($form->has('numero_elementi')) {
            $this->limit = $form->get('numero_elementi')->getData();
        }
        return $this->limit;
    }
}
