<?php

namespace Schema31\UtilityBundle\Service\Ricerca;

use Doctrine\Common\Annotations\Reader;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Schema31\UtilityBundle\Service\IRicerca;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

/**
 * Servizio che cerca di standardizare la ricerca nel sistema
 * Il servizio ha un metodo ricerca che genera la form di ricerca,effettua la query e ritorna i risultati
 */
class Post implements IRicerca {
    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;
    /**
     * @var SessionInterface
     */
    protected $session;

    /**
     * @var PaginatorInterface
     */
    protected $paginator;
    /**
     * @var Request
     */
    protected $request;
    /**
     * @var FormFactoryInterface
     */
    protected $formFactory;

    /**
     * @var object|array|null
     */
    protected $data;

    /**
     * @var int[]
     */
    protected $listaElementiPerPagina = [
        '5' => 5,
        '10' => 10,
        '25' => 25,
        '50' => 50,
        '75' => 75,
        '100' => 100,
    ];
    /**
     * @var int
     */
    protected $numeroMassimoElementi;
    /**
     * @var int
     */
    protected $numeroDefaultElementi;

    /**
     * @var object|array|null
     */
    protected $prototype;

    /**
     * @var Reader
     */
    protected $annotationReader;

    /**
     * @var int
     */
    protected $limit;

    public function __construct(
            $numeroElementi,
            EntityManagerInterface $entityManager,
            SessionInterface $session,
            PaginatorInterface $paginator,
            RequestStack $requestStack,
            FormFactoryInterface $formFactory,
            Reader $annotationReader
        ) {
        $this->entityManager = $entityManager;
        $this->session = $session;
        $this->paginator = $paginator;
        $this->request = $requestStack->getCurrentRequest();
        $this->formFactory = $formFactory;
        $this->numeroDefaultElementi = $numeroElementi;
        $this->annotationReader = $annotationReader;
    }

    public function pulisci(array $options): void {
        $parameters = $this->resolveOptions($options);
        $nomeOggettoInSessione = $this->getNomeOggettoInSessione();
        $this->session->remove($nomeOggettoInSessione);
        $this->session->remove($nomeOggettoInSessione . '_elements');
    }

    public function ricerca(array $options = []): array {
        $parameters = $this->resolveOptions($options);
        $nomeOggettoInSession = $this->getNomeOggettoInSessione();

        $this->data = $this->session->get($nomeOggettoInSession, $this->prototype);
        if ($this->session->has($nomeOggettoInSession)) {
            $this->rendeManaged();
        }

        $this->limit = $this->getNumeroElementiPerPagina();
        $listaElementiPerPagina = $parameters['choice_elements_per_page'];
        if (!\in_array($this->limit, $listaElementiPerPagina)) {
            $listaElementiPerPagina["{$this->limit}"] = $this->limit;
            \asort($listaElementiPerPagina);
        }

        $formRicerca = $this->formFactory->create($options['form_type'], $this->data);

        if ($parameters['show_elements_number']) {
            $formRicerca->add('numero_elementi', ChoiceType::class, [
                'required' => false,
                'choices' => $this->listaElementiPerPagina,
                'placeholder' => false,
                'label' => 'Elementi per pagina',
                'mapped' => false,
            ]);
        }

        $nomeParametroPagina = $parameters['page_index'];
        $numeroPaginaRicerca = $this->request->get($nomeParametroPagina, 1);
        $formRicerca->handleRequest($this->request);
        if ($formRicerca->isSubmitted()) {
            $this->data = $formRicerca->getData();
            $this->session->set($nomeOggettoInSession, $this->data);
            $numeroPaginaRicerca = 1;
            $this->limit = $formRicerca->has('numero_elementi') ? $formRicerca->get('numero_elementi')->getData() : $this->limit;
            $this->session->set($nomeOggettoInSession . '_elements', $this->limit);
        }

        $filtroAttivo = $this->session->has($nomeOggettoInSession);

        $pagination_object = [];
        if ($filtroAttivo || true === $parameters['empty_search']) {
            $repository = $this->entityManager->getRepository($parameters['repository']);
            $pagination_object = call_user_func([$repository, $parameters['repository_method']], $this->data);
        }

        $pagination = $this->paginator->paginate(
            $pagination_object,
            $numeroPaginaRicerca,
            $this->limit
        );

        return [
            "risultato" => $pagination,
            "form_ricerca" => $formRicerca->createView(),
            "filtro_attivo" => $filtroAttivo,
        ];
    }

    private function resolveOptions(array $options): array {
        $resolver = new OptionsResolver();
        $resolver->setDefaults([
            'data' => null,
            'choice_elements_per_page' => [
                '5' => 5,
                '10' => 10,
                '25' => 25,
                '50' => 50,
                '75' => 75,
                '100' => 100,
            ],
            'elements_per_page' => $this->numeroDefaultElementi,
            'page_index' => 'page',
            'show_elements_number' => false,
            'empty_search' => true,
        ]);
        $resolver->setRequired('form_type');
        $resolver->setRequired('repository');
        $resolver->setRequired('repository_method');

        $resolver->setAllowedTypes('form_type', 'string');
        $resolver->setAllowedTypes('repository', 'string');
        $resolver->setAllowedTypes('repository_method', 'string');
        $resolver->setAllowedTypes('page_index', 'string');
        $resolver->setAllowedTypes('choice_elements_per_page', 'int[]');
        $resolver->setAllowedTypes('elements_per_page', 'int');
        $resolver->setAllowedTypes('show_elements_number', 'bool');
        $resolver->setAllowedTypes('empty_search', 'bool');

        $parameters = $resolver->resolve($options);

        $this->prototype = $parameters['data'];

        return $parameters;
    }

    /** nel caso si ripeschi un oggetto dalla sessione le entity sono tutte detached
     * quindi nel caso si ricerchi usando oggetti di tipo Entity cerco di farne il merge
     * in caso di errore del tipo "Entities passed to the choice field must be managed"
     * controlla che la base class non abbia attributi private
     */
    protected function rendeManaged(): void {
        $modelloReflection = new \ReflectionClass($this->data);
        $currentProps = $modelloReflection->getProperties();
        $parent = $modelloReflection->getParentClass();
        $parentProps = $parent ? $parent->getProperties() : [];
        $props = \array_merge($currentProps, $parentProps);

        foreach ($props as $prop) {
            $prop->setAccessible(true);
            $valore = $prop->getValue($this->data);

            if (!\is_object($valore) || $valore instanceof \DateTime) {
                continue;
            }

            $new_valore = $this->getNewValore($valore);
            if ($valore instanceof Collection) {
                $new_valore = new ArrayCollection();
                foreach ($valore as $val) {
                    $new_valore_elemento = $this->getNewValore($val);
                    $new_valore->add($new_valore_elemento);
                }
            }
            $prop->setValue($this->data, $new_valore);
        }
    }

    protected function getNewValore(object $valore) {
        $proxy_class_name = \get_class($valore);
        $class_name = $this->entityManager->getClassMetadata($proxy_class_name)->rootEntityName;
        $new_valore = $this->getValore($class_name, $valore);

        return $new_valore;
    }

    protected function getValore(string $class_name, \object $valore) {
        $propReflection = new \ReflectionClass($class_name);
        $annotazione_entity = $this->annotationReader->getClassAnnotation($propReflection, "Doctrine\ORM\Mapping\Entity");
        if ($annotazione_entity) {
            if (method_exists($valore, "getId") && !\is_null($valore->getId())) {
                $new_valore = $this->entityManager->getRepository(\get_class($valore))->find($valore->getId());
            } else {
                $new_valore = $this->entityManager->merge($valore);
            }
            return $new_valore;
        }
        return null;
    }

    private function getNumeroElementiPerPagina(): int {
        $nomeOggettoInSession = $this->getNomeOggettoInSessione() . '_elements';
        $valoreInSessione = $this->session->get($nomeOggettoInSession, null);
        $numElementi = $this->limit ??
                $valoreInSessione ??
                $this->numeroDefaultElementi ??
                0;

        return $numElementi;
    }

    private function getNomeOggettoInSessione(): string {
        $modelloReflection = new \ReflectionClass($this->prototype);
        return $modelloReflection->getName();
    }
}
