<?php

namespace Schema31\UtilityBundle\Service\Ricerca;

use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Schema31\UtilityBundle\Service\IRicerca;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\OptionsResolver\OptionsResolver;

/**
 * Servizio che cerca di standardizare la ricerca nel sistema
 * Il servizio ha un metodo ricerca che genera la form di ricerca,effettua la query e ritorna i risultati
 */
class Get implements IRicerca {
    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;

    /**
     * @var PaginatorInterface
     */
    protected $paginator;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var FormFactoryInterface
     */
    protected $formFactory;

    protected $data;

    /**
     * Numero default elementi per pagina
     * @var int
     */
    protected $numeroDefaultElementi;

    /**
     * Numero elementi selezionato dal form di ricerca
     * @var int|null
     */
    protected $limit;

    public function __construct(
            $numeroElementi,
            EntityManagerInterface $entityManager,
            PaginatorInterface $paginator,
            RequestStack $requestStack,
            FormFactoryInterface $formFactory
        ) {
        $this->entityManager = $entityManager;
        $this->paginator = $paginator;
        $this->request = $requestStack->getCurrentRequest();
        $this->formFactory = $formFactory;
        $this->numeroDefaultElementi = $numeroElementi;
    }

    public function pulisci(array $attributiRicerca): void {
    }

    public function ricerca(array $options = []): array {
        $parameters = $this->resolveOptions($options);
        $this->data = $parameters['data'];

        $this->limit = $this->getNumeroElementiPerPagina();
        $listaElementiPerPagina = $parameters['choice_elements_per_page'];
        if (!\in_array($this->limit, $listaElementiPerPagina)) {
            $listaElementiPerPagina["{$this->limit}"] = $this->limit;
            \asort($listaElementiPerPagina);
        }

        $formRicerca = $this->formFactory
            ->createBuilder($options['form_type'], $this->data)
            ->setMethod('GET')
            ->getForm();

        if ($parameters['show_elements_number']) {
            $formRicerca->add('numero_elementi', ChoiceType::class, [
                'required' => false,
                'choices' => $listaElementiPerPagina,
                'placeholder' => false,
                'label' => 'Elementi per pagina',
                'mapped' => false,
            ]);
        }

        $formRicerca->handleRequest($this->request);
        $filtroAttivo = false;
        if ($formRicerca->isSubmitted()) {
            $filtroAttivo = true;
            $this->limit = $formRicerca->has('numero_elementi') ? $formRicerca->get('numero_elementi')->getData() : $this->limit;
        }

        $repository = $this->entityManager->getRepository($parameters['repository']);
        $paginationObject = \call_user_func([$repository, $parameters['repository_method']], $this->data);
        $nomeParametroPagina = $parameters['page_index'];
        $numeroPaginaRicerca = $this->request->get($nomeParametroPagina, 1);
        $pagination = $this->paginator->paginate(
            $paginationObject,
            $numeroPaginaRicerca,
            $this->limit
        );

        return [
            "risultato" => $pagination,
            "form_ricerca" => $formRicerca->createView(),
            "filtro_attivo" => $filtroAttivo,
        ];
    }

    private function resolveOptions(array $options): array {
        $resolver = new OptionsResolver();
        $resolver->setDefaults([
            'data' => null,
            'choice_elements_per_page' => [
                '5' => 5,
                '10' => 10,
                '25' => 25,
                '50' => 50,
                '75' => 75,
                '100' => 100,
            ],
            'elements_per_page' => $this->numeroDefaultElementi,
            'page_index' => 'page',
            'show_elements_number' => false,
        ]);
        $resolver->setRequired('form_type');
        $resolver->setRequired('repository');
        $resolver->setRequired('repository_method');

        $resolver->setAllowedTypes('form_type', 'string');
        $resolver->setAllowedTypes('repository', 'string');
        $resolver->setAllowedTypes('repository_method', 'string');
        $resolver->setAllowedTypes('page_index', 'string');
        $resolver->setAllowedTypes('choice_elements_per_page', 'int[]');
        $resolver->setAllowedTypes('elements_per_page', 'int');
        $resolver->setAllowedTypes('show_elements_number', 'bool');

        $parameters = $resolver->resolve($options);

        return $parameters;
    }

    private function getNumeroElementiPerPagina(): int {
        $numElementi = $this->limit ??
                        $this->numeroDefaultElementi ??
                        0;

        return $numElementi;
    }
}
