<?php

namespace Schema31\UtilityBundle\Service\Ricerca;

use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Schema31\UtilityBundle\Model\IAttributiRicerca;
use Schema31\UtilityBundle\Service\IRicerca;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Servizio che cerca di standardizare la ricerca nel sistema
 * Il servizio ha un metodo ricerca che genera la form di ricerca,effettua la query e ritorna i risultati
 */
class Get implements IRicerca {
    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;

    /**
     * @var PaginatorInterface
     */
    protected $paginator;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var FormFactoryInterface
     */
    protected $formFactory;

    /**
     * @var IAttributiRicerca
     */
    protected $data;

    /**
     * @var int[]
     */
    protected $listaElementiPerPagina = [
        '5' => 5,
        '10' => 10,
        '25' => 25,
        '50' => 50,
        '75' => 75,
        '100' => 100,
    ];
    /**
     * @var int
     */
    protected $numeroMassimoElementi;
    /**
     * @var int
     */
    protected $numeroElementi;

    public function __construct(
            $numeroElementi,
            EntityManagerInterface $entityManager,
            PaginatorInterface $paginator,
            RequestStack $requestStack,
            FormFactoryInterface $formFactory
        ) {
        $this->entityManager = $entityManager;
        $this->paginator = $paginator;
        $this->request = $requestStack->getCurrentRequest();
        $this->formFactory = $formFactory;
        $this->numeroElementi = $numeroElementi;
    }

    public function pulisci(IAttributiRicerca $attributiRicerca): void {
    }

    public function ricerca(IAttributiRicerca $datiRicerca, array $options = []): array {
        $this->data = $datiRicerca;
        $numElementi = $this->getNumeroElementiPerPagina();
        if (!\in_array($numElementi, $this->listaElementiPerPagina)) {
            $this->listaElementiPerPagina["$numElementi"] = $numElementi;
            \asort($this->listaElementiPerPagina);
        }
        $this->data->setNumeroElementi($numElementi);

        $typeReflection = new \ReflectionClass($this->data->getType());
        $formTypeString = $typeReflection->getName();
        $formRicerca = $this->formFactory
            ->createBuilder($formTypeString, $this->data, $options)
            ->setMethod('GET')
            ->getForm();

        if ($this->data->mostraNumeroElementi()) {
            $formRicerca->add('numero_elementi', ChoiceType::class, [
                'required' => false,
                'choices' => $this->listaElementiPerPagina,
                'placeholder' => false,
                'label' => 'Elementi per pagina',
            ]);
        }

        $formRicerca->handleRequest($this->request);
        $filtroAttivo = $formRicerca->isSubmitted();
        $this->data->setFiltroAttivo($filtroAttivo);

        $repository = $this->entityManager->getRepository($this->data->getNomeRepository());
        $paginationObject = \call_user_func([$repository, $this->data->getNomeMetodoRepository()], $this->data);
        $nomeParametroPagina = $this->data->getNomeParametroPagina();
        $nomeParametroPagina = $nomeParametroPagina ?? "page";
        $numeroPaginaRicerca = $this->request->get($nomeParametroPagina, 1);
        $pagination = $this->paginator->paginate(
            $paginationObject,
            $numeroPaginaRicerca,
            $this->data->getNumeroElementi()
        );

        return [
            "risultato" => $pagination,
            "form_ricerca" => $formRicerca->createView(),
            "filtro_attivo" => $filtroAttivo,
        ];
    }

    private function getNumeroElementiPerPagina(): int {
        $numeroElementiDefault = $this->numeroElementi;
        $numeroElementiPerPagina = $this->data->getNumeroElementiPerPagina();
        $numElementi = $this->data->getNumeroElementi();
        $numElementi = $numElementi ??
                $numeroElementiPerPagina ??
                $numeroElementiDefault ??
                0;

        return $numElementi;
    }
}
