<?php

namespace Schema31\UtilityBundle\Service;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Schema31\UtilityBundle\Model\IAttributiRicerca;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Servizio che cerca di standardizare la ricerca nel sistema
 * Il servizio ha un metodo ricerca che genera la form di ricerca,effettua la query e ritorna i risultati
 */
class RicercaService {
    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;
    /**
     * @var SessionInterface
     */
    protected $session;

    /**
     * @var PaginatorInterface
     */
    protected $paginator;
    /**
     * @var Request
     */
    protected $request;
    /**
     * @var FormFactoryInterface
     */
    protected $formFactory;

    /**
     * @var IAttributiRicerca
     */
    protected $data;

    /**
     * @var int[]
     */
    protected $listaElementiPerPagina = [
        '5' => 5,
        '10' => 10,
        '25' => 25,
        '50' => 50,
        '75' => 75,
        '100' => 100,
    ];
    /**
     * @var int
     */
    protected $numeroMassimoElementi;
    /**
     * @var int
     */
    protected $numeroElementi;

    public function __construct(
            $numeroMassimoElementi,
            $numeroElementi,
            EntityManagerInterface $entityManager,
            SessionInterface $session,
            PaginatorInterface $paginator,
            RequestStack $requestStack,
            FormFactoryInterface $formFactory
        ) {
        $this->entityManager = $entityManager;
        $this->session = $session;
        $this->paginator = $paginator;
        $this->request = $requestStack->getCurrentRequest();
        $this->formFactory = $formFactory;
        $this->numeroMassimoElementi = $numeroMassimoElementi;
        $this->numeroElementi = $numeroElementi;
    }

    public function pulisci(IAttributiRicerca $attributiRicerca): void {
        $this->session->remove($this->getNomeOggettoInSessione($attributiRicerca));
    }

    public function ricerca(IAttributiRicerca $datiRicerca, array $options = []): array {
        $nomeOggettoInSession = $this->getNomeOggettoInSessione($datiRicerca);
        $typeReflection = new \ReflectionClass($datiRicerca->getType());

        $this->data = $this->session->get($nomeOggettoInSession, $datiRicerca);
        if ($this->session->has($nomeOggettoInSession)) {
            $this->rendeManaged();
        }

        $this->data->mergeFreshData($datiRicerca);
        $numElementi = $this->getNumeroElementiPerPagina($this->data);
        if (!in_array($numElementi, $this->listaElementiPerPagina)) {
            $this->listaElementiPerPagina["$numElementi"] = $numElementi;
            asort($this->listaElementiPerPagina);
        }

        $this->data->setNumeroElementi($numElementi);

        $formTypeString = $typeReflection->getName();
        $formRicerca = $this->formFactory->create($formTypeString, $this->data, $options);

        if ($this->data->mostraNumeroElementi()) {
            $formRicerca->add('numero_elementi', ChoiceType::class, [
                'required' => false,
                'choices' => $this->listaElementiPerPagina,
                'placeholder' => false,
                'label' => 'Elementi per pagina',
            ]);
        }

        $nomeParametroPagina = $datiRicerca->getNomeParametroPagina();
        $nomeParametroPagina = $nomeParametroPagina ?? "page";
        $numeroPaginaRicerca = $this->request->attributes->getInt($nomeParametroPagina, 1);
        $formRicerca->handleRequest($this->request);
        if ($formRicerca->isSubmitted()) {
            $this->data = $formRicerca->getData();
            $this->session->set($nomeOggettoInSession, $this->data);
            $numeroPaginaRicerca = 1;
        }

        $ricercaVuota = $this->data->isRicercaVuota();

        $filtroAttivo = $this->session->has($nomeOggettoInSession);
        $this->data->setFiltroAttivo($filtroAttivo);

        if ($ricercaVuota && false === $this->data->getConsentiRicercaVuota()) {
            $pagination_object = [];
        } else {
            $repository = $this->entityManager->getRepository($datiRicerca->getNomeRepository());
            $pagination_object = call_user_func([$repository, $datiRicerca->getNomeMetodoRepository()], $this->data);
        }

        $pagination = $this->paginator->paginate(
            $pagination_object,
            $numeroPaginaRicerca,
            $this->data->getNumeroElementi()
        );

        return [
            "risultato" => $pagination,
            "form_ricerca" => $formRicerca->createView(),
            "filtro_attivo" => $filtroAttivo,
        ];
    }

    /** nel caso si ripeschi un oggetto dalla sessione le entity sono tutte detached
     * quindi nel caso si ricerchi usando oggetti di tipo Entity cerco di farne il merge
     * in caso di errore del tipo "Entities passed to the choice field must be managed"
     * controlla che la base class non abbia attributi private
     */
    protected function rendeManaged(): void {
        $modelloReflection = new \ReflectionClass($this->data);
        $currentProps = $modelloReflection->getProperties();
        $parent = $modelloReflection->getParentClass();
        $parentProps = $parent ? $parent->getProperties() : [];
        $props = \array_merge($currentProps, $parentProps);

        foreach ($props as $prop) {
            $prop->setAccessible(true);
            $valore = $prop->getValue($this->data);

            if (!\is_object($valore) || $valore instanceof \DateTime) {
                continue;
            }

            $new_valore = $this->getNewValore($valore);
            if ($valore instanceof Collection) {
                $new_valore = new ArrayCollection();
                foreach ($valore as $val) {
                    $new_valore_elemento = $this->getNewValore($val);
                    if (!\is_null($new_valore)) {
                        $new_valore->add($new_valore_elemento);
                    }
                }
            }
            $prop->setValue($this->data, $new_valore);
        }
    }

    protected function getNewValore($valore) {
        $proxy_class_name = \get_class($valore);
        $class_name = $this->entityManager->getClassMetadata($proxy_class_name)->rootEntityName;
        $new_valore = $this->getValore($class_name, $valore);

        return $new_valore;
    }

    protected function getValore(string $class_name, $valore) {
        $annotation_reader = $this->container->get("annotation_reader");
        $propReflection = new \ReflectionClass($class_name);
        $annotazione_entity = $annotation_reader->getClassAnnotation($propReflection, "Doctrine\ORM\Mapping\Entity");
        if ($annotazione_entity) {
            if (method_exists($valore, "getId") && !\is_null($valore->getId())) {
                $new_valore = $this->entityManager->getRepository(\get_class($valore))->find($valore->getId());
            } else {
                $new_valore = $this->entityManager->merge($valore);
            }
            return $new_valore;
        }
        return null;
    }

    private function getNumeroElementiPerPagina($datiRicerca): int {
        $numeroElementiMassimo = $this->numeroMassimoElementi;
        $numeroElementiDefault = $this->numeroElementi;
        $numeroElementiPerPagina = $datiRicerca->getNumeroElementiPerPagina();
        $numElementi = $datiRicerca->getNumeroElementi();
        $numElementi = $numElementi ??
                $numeroElementiPerPagina ??
                $numeroElementiDefault ??
                0;

        $numElementi = !$datiRicerca->getBypassMaxElementiPerPagina() &&
            $numElementi >= $numeroElementiMassimo ?
                $numeroElementiMassimo :
                $numElementi;

        return $numElementi;
    }

    private function getNomeOggettoInSessione(IAttributiRicerca $attributiRicerca): string {
        $modelloReflection = new \ReflectionClass($attributiRicerca);
        return $modelloReflection->getName();
    }
}
