<?php

namespace Schema31\SharepointStorage\Service;

use Office365\PHP\Client\Runtime\Auth\AuthenticationContext;
use Office365\PHP\Client\SharePoint\ClientContext;

use Schema31\Storage\Service\StorageInterface;

/**
 * Description of Sharepoint
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class Sharepoint implements StorageInterface {
    
    private $clientContext;
    private $sitePath;
    private $siteUrl;
    private $folderPath;
    
    public function __construct(string $authentationUrl, string $username, string $password, string $sitePath, ?string $folderPath = NULL) {
        
        $authCtx = new AuthenticationContext($authentationUrl);
        $authCtx->acquireTokenForUser($username, $password);
        
        $this->sitePath = trim($sitePath, "/");
        $this->folderPath = trim($folderPath, "/");
        $this->siteUrl = $authentationUrl."/".$this->sitePath;
        $this->clientContext = new ClientContext($this->siteUrl, $authCtx);
    }
    
    private function getFileObject(string $fileKey) {
        $serializationContext = new \Office365\PHP\Client\Runtime\OData\JsonLightSerializerContext(\Office365\PHP\Client\Runtime\OData\ODataMetadataLevel::Verbose);
        
        $url = $this->clientContext->getServiceRootUrl() . "web/getfilebyid('$fileKey')";
        $options = new \Office365\PHP\Client\Runtime\Utilities\RequestOptions($url);
        $json = $this->clientContext->executeQueryDirect($options);
        
        $file = new \Office365\PHP\Client\SharePoint\File($this->clientContext);
        $result = new \Office365\PHP\Client\Runtime\ClientResult($file);
        $result->fromJson(json_decode($json), $serializationContext);  
        
        return $file;
    }
    
    public function deleteFile(string $fileKey, ?int $fileVersion = null) {
        if ($fileVersion !== NULL) {
            return FALSE;
        }
        
        try { 
            $file = $this->getFileObject($fileKey);
            $file->deleteObject();
            $this->clientContext->executeQuery();
        } catch (\Exception $e) {
            return FALSE;
        }
    }

    public function detailFile(string $fileKey, ?int $fileVersion = null) {
        if ($fileVersion !== NULL) {
            return FALSE;
        }
        
        try { 
            $file = $this->getFileObject($fileKey);

            $creationTime = new \DateTime($file->getProperty("TimeCreated"));
            
            $details = new \stdClass();
            $details->isDeleted = $file->getProperty("Exists") ? "0" : "1";
            $details->fileKey = $file->getProperty("UniqueId");
            $details->fileVersion = $file->getProperty("MajorVersion").$file->getProperty("MinorVersion");
            $details->creationTime = $creationTime->format("Y-m-d H:i:s");
            $details->fileSize = $file->getProperty("Length");
            $details->fileName = $file->getProperty("Title");

            // mancano fileMimeType, fsFileId, fileMD5, fileSHA256, Repository, publicRepository, friendlyUrl
            return $details;
        } catch (\Exception $e) {
            return FALSE;
        }        
    }

    public function getFile(string $fileKey, ?int $fileVersion = null) {       
        if ($fileVersion !== NULL) {
            return FALSE;
        }
        
        try {            
            $url = $this->clientContext->getServiceRootUrl() . "web/getfilebyid('$fileKey')/\$value";
            $options = new \Office365\PHP\Client\Runtime\Utilities\RequestOptions($url);

            return $this->clientContext->executeQueryDirect($options);
        } catch (\Exception $e) {
            return FALSE;
        }
    }

    public function sendFile(string $localFile, ?string $mime = null, ?string $publicName = NULL, ?string $fileKey = NULL) {
        try {
            if ($fileKey !== NULL) {
                return FALSE;
            }
            
            if (!file_exists($localFile)) {
                return FALSE;
            }
            
            if (is_null($publicName)) {
                $publicName = basename($localFile);
            }
            
            if (is_null($mime)) {
                $mime = mime_content_type($localFile);
            }
            
            $folderByServerRelativeUrl = $this->calculateFolderByServerRelativeUrl();
            $fileCreationInformation = new \Office365\PHP\Client\SharePoint\FileCreationInformation();
            $fileCreationInformation->Content = file_get_contents($localFile);
            $fileCreationInformation->Url = $this->uniqidReal();
            $uploadFile = $this->clientContext->getWeb()->getFolderByServerRelativeUrl($folderByServerRelativeUrl)->getFiles()->add($fileCreationInformation);
            $this->clientContext->executeQuery();
            
            $uploadFile->getListItemAllFields()->setProperty("Title", $publicName);    
            $uploadFile->getListItemAllFields()->update();
            $this->clientContext->executeQuery();
            return $uploadFile->getProperty("UniqueId");
        } catch (\Exception $e) {
            return FALSE;
        }        
    }

    public function streamFile(string $fileKey, ?int $fileVersion = null): bool {
        if ($fileVersion !== NULL) {
            return FALSE;
        }
        
        try {         
            $details = $this->detailFile($fileKey, $fileVersion);
            $content = $this->getFile($fileKey, $fileVersion);
            
            if (isset($details->fileMimeType)) {
                header("Content-Type: {$details->fileMimeType}");
            }
            
            $length = isset($details->fileSize) ? $details->fileSize : strlen($content);
            header("Content-Length: $length");
            
            header('Content-Disposition: inline; filename="'.$details->fileName.'"');
            
            echo $content;
            
            return TRUE;
        } catch (\Exception $e) {
            return FALSE;
        }     
    }
    
//    public function getFolderOrCreateIfNotExists($folderUrl) {
//        try {
//            $folderByServerRelativeUrl = $this->sitePath.$folderUrl;
//            $folder = $this->clientContext->getWeb()->getFolderByServerRelativeUrl($folderByServerRelativeUrl);
//            
//            if ($folder->getServerObjectIsNull()) {
//                $this->clientContext->load($folder);
//                $this->clientContext->executeQuery();
//            }
//            
//            return $folder;
//        } catch (\Exception $e) {
//            $a = 1;
//        }
//    }
    
    protected function calculateFolderByServerRelativeUrl() {
        return "/".$this->sitePath."/".$this->folderPath;
    }
 
    private function uniqidReal($lenght = 13) {
        // uniqid gives 13 chars, but you could adjust it to your needs.
        if (function_exists("random_bytes")) {
            $bytes = random_bytes(ceil($lenght / 2));
        } elseif (function_exists("openssl_random_pseudo_bytes")) {
            $bytes = openssl_random_pseudo_bytes(ceil($lenght / 2));
        } else {
            throw new Exception("no cryptographically secure random function available");
        }
        return substr(bin2hex($bytes), 0, $lenght);
    }    
}
