<?php

namespace Schema31\SafeHubLib\Service;

use Schema31\Exceptions\Operation\UnexpectedException;

use Schema31\SafeHubLib\Service\EntityManager;
use Schema31\SafeHubLib\Service\DocumentService;

/**
 * Description of StorageService
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class StorageService {
    private $em;
    private $documentService;
    private $providers;
    
    public function __construct(EntityManager $entityManager, DocumentService $documentService) {
        $this->em = $entityManager;
        $this->documentService = $documentService;
    }
    
    public function registerProvider($providerName, $providerInstance) {
        $this->providers[$providerName] = $providerInstance;
    }
	
    public function getFileShortUrl($documentId, $userId, $path) {
        $rawFileContainer = $this->getRawFileContainer($documentId, $userId, $path);
        
        $provider = $this->getProvider($rawFileContainer);
        
        return $this->providers[$provider]->getShortUrl($rawFileContainer->RawFile);      
    }
    
    public function getFile($documentId, $userId, $path) {
        $rawFileContainer = $this->getRawFileContainer($documentId, $userId, $path);
        
        $provider = $this->getProvider($rawFileContainer);
        
        return $this->providers[$provider]->getFile($rawFileContainer->RawFile);      
    }    
    
    private function getRawFileContainer($documentId, $userId, $path) {
        $document = $this->em->getRepository("Base")->getById($documentId);
        
        $this->documentService->checkAccessByUserId($document, $userId);
        
        $pathParts = explode("/", $path);
        
        $rawFileContainer = array_reduce($pathParts, function($subDocument, $pathPart) {
            if (is_null($subDocument)) {
                return null;
            }
            
            if (is_object($subDocument) && isset($subDocument->$pathPart)) {
                return $subDocument->$pathPart;
            }
            
            if (is_array($subDocument) && isset($subDocument[$pathPart])) {
                return $subDocument[$pathPart];
            } 
            
            return null;
        }, $document);
        
        if (!isset($rawFileContainer->RawFile)) {
            throw new UnexpectedException("RawFile not found in document ".$documentId);
        }
        
        if (!isset($rawFileContainer->RawFile->Provider)) {
            throw new UnexpectedException("Storage provider not found");
        }

        return $rawFileContainer;
    }
    
    private function getProvider($rawFileContainer) {
        $provider = strtolower($rawFileContainer->RawFile->Provider);
        
        if (!isset($this->providers[$provider])) {
            throw new UnexpectedException("Storage provider not supported");
        }

        return $provider;
    }
}
