<?php

namespace Schema31\SafeHubLib\Service;

use Schema31\SafeHubLib\IdentityProvider\SafeIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\GoogleIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\FacebookIdentityProvider;
use Schema31\SafeHubLib\Exception\ValidationException;

use Schema31\SafeHubLib\Service\EntityManager;
use Twig\Environment;
use Symfony\Component\Translation\TranslatorInterface;
use PHPMailer\PHPMailer\PHPMailer;

use Schema31\CouchDBClient\Service\CouchDBClient;

/**
 * Description of UserService
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class UserService extends PhysicalEntityService {

    const IDENTITY_NOT_VALID_ERROR = "U-0002";
    const IDENTITY_ALREADY_PRESENT_ERROR = "U-0011";
    const LOGIN_IDENTITY_NOT_FOUND = "U-0013";
    const USER_NOT_FOUND = "U-0010";
    const DOCUMENT_TEMPLATE = 1;
    
    const DEVICE_ID_NOT_VALID = "U-0014";
    const DEVICE_TYPE_NOT_VALID = "U-0015";
    
    const PHYSICAL_ENTITY = "PhysicalEntity";
    const LEGAL_ENTITY = "LegalEntity";

    private $identityProviders;
    private $em;
    private $translator;
    private $templating;
    private $mailer;
    
    private $userCouchDBClient;
    private $userCouchDBProtocol;
    private $userCouchDBHost;
    private $userCouchDBPort;
    
    private $replicationCouchDBClient;
    private $appCouchDBHost;
    private $appCouchDBName;
    private $appCouchDBUsername;
    private $appCouchDBPassword;
    
    public function __construct(EntityManager $entityManager, Environment $templating, TranslatorInterface $translator, PHPMailer $mailer) {
        $this->identityProviders = [
            new SafeIdentityProvider(),
            new GoogleIdentityProvider(),
            new FacebookIdentityProvider()
        ];

        $this->em = $entityManager;

        $this->translator = $translator;
        $this->templating = $templating;
        $this->mailer = $mailer;
    }
    
    public function setUserDBParameters(CouchDBClient $userCouchDBClient, $userCouchDBProtocol, $userCouchDBHost, $userCouchDBPort) {
        $this->userCouchDBClient = $userCouchDBClient;
        $this->userCouchDBProtocol = $userCouchDBProtocol;
        $this->userCouchDBHost = $userCouchDBHost;
        $this->userCouchDBPort = $userCouchDBPort;
    }
    
    public function setReplicationDBParameters(CouchDBClient $replicationCouchDBClient, $appCouchDBHost, $appCouchDBName, $appCouchDBUsername, $appCouchDBPassword) {
        $this->replicationCouchDBClient = $replicationCouchDBClient;
        $this->appCouchDBHost = $appCouchDBHost;
        $this->appCouchDBName = $appCouchDBName;
        $this->appCouchDBUsername = $appCouchDBUsername;
        $this->appCouchDBPassword = $appCouchDBPassword;        
    }

    /**
     * Create a basic user document
     * 
     * @param string $fiscalCode
     * @return \stdClass
     */
    public function createUserDocument(string $fiscalCode) {
        $physicalEntity = new \stdClass();
        $physicalEntity->_id = $this->em->getRepository(self::PHYSICAL_ENTITY)->generateId($fiscalCode);
        $physicalEntity->Type = "PhysicalEntity";
        $physicalEntity->DocumentTemplate = self::DOCUMENT_TEMPLATE;
        $physicalEntity->FiscalCode = $fiscalCode;

        return $physicalEntity;
    }

    public function createUser(string $firstName, string $lastName, string $fiscalCode, string $email, \stdClass $identity, string $language, bool $createUserDB = false) {
        $identityProvider = $this->getProvider($identity);

        // validate the identity
        $identityProvider->validate($identity);

        // normalize the identity
        $identityProvider->normalize($identity);

        $providerKey = $identityProvider->getProviderKey();

        // look for existing physical entities with the provided identity
        $userByIdentity = $this->em->getRepository(self::PHYSICAL_ENTITY)->findOneByIdentity($providerKey, $identity->$providerKey->LoginKey);

        if ($userByIdentity) {
            throw new ValidationException(self::IDENTITY_ALREADY_PRESENT_ERROR, "Provided identity already present");
        }

        // look for existing physical entities with the provided fiscal code
        $physicalEntity = $this->em->getRepository(self::PHYSICAL_ENTITY)->findOneByFiscalCode($fiscalCode);

        // if exists
        if ($physicalEntity) {
            // and is already a user and has the provided identity
            if (isset($physicalEntity->Identities->$providerKey)) {
                // is really needed this check?
                throw new ValidationException(self::IDENTITY_ALREADY_PRESENT_ERROR, "Provided identity already present");
            }

            $identity->$providerKey->VerificationCode = substr(bin2hex(random_bytes(5)), 0, 4);
        } else {
            // physical entity creation
            $physicalEntity = $this->createUserDocument($fiscalCode);
            $physicalEntity->FirstName = $firstName;
            $physicalEntity->LastName = $lastName;
        }
        
        $this->validate($physicalEntity);

        $contacts = new \stdClass();
        $contacts->Email = new \stdClass();
        $contacts->Email->Value = $email;
        $contacts->Email->Verified = $identityProvider->isEmailVerified();
        if (!$contacts->Email->Verified) {
            $contacts->Email->VerificationCode = substr(bin2hex(random_bytes(5)), 0, 4);
        }
        $physicalEntity->Contacts = $contacts;

        $preferences = new \stdClass();
        $preferences->Language = $language;
        $physicalEntity->Preferences = $preferences;

        $identityProvider->finalize($identity);

        // add identity
        $this->addIdentity($physicalEntity, $identity);

        try {
            // aggiungiamo la rappresentanza verso se stesso
            $this->addUserLegalEntity($physicalEntity);
        } catch (\Exception $ex) {
            // eventuali errori non bloccanti per la registrazione
        }

        $this->em->getRepository(self::PHYSICAL_ENTITY)->save($physicalEntity);

        // aggiunta db, se necessario
        if ($createUserDB) {
            try {
                $this->createDatabaseUserInfo($physicalEntity);
            } catch (\Exception $ex) {
                // eventuali errori non bloccanti per la registrazione
            }
        }
        
        // proviamo ad aggiungere il documento per le notifiche, se necessario
        try {
            $this->addNotificationsDoc($physicalEntity);
        } catch (\Exception $ex) {
            // eventuali errori non bloccanti per la registrazione
        }

        $this->sendRegistrationEmail($physicalEntity);

        return $this->_reduceDocument($physicalEntity);
    }
    
    public function addNotificationsDoc($physicalEntity) {
        $notificationDoc = $this->em->getRepository(self::PHYSICAL_ENTITY)->findNotificationsDoc($physicalEntity->FiscalCode);
        if (is_null($notificationDoc)) {
            $notificationDoc = new \stdClass();
            $notificationDoc->_id = $this->em->getRepository(self::PHYSICAL_ENTITY)->generateNotificationsDocId($physicalEntity->FiscalCode);
            $notificationDoc->Type = "Notifications";
            
            $welcomeNotification = new \stdClass();
            $welcomeNotification->Date = $physicalEntity->CreationDate ?: (new \DateTime())->format(\DateTime::ATOM);
            $welcomeNotification->Title = new \stdClass();
            $welcomeNotification->ShortContent = new \stdClass();
            foreach(["it", "en"] as $language) {
                $welcomeNotification->Title->$language = $this->translator->trans("welcome_notification_title", [], null, $language);                
                $welcomeNotification->ShortContent->$language = $this->translator->trans("welcome_notification_short_content", [], null, $language);                
            }
            $welcomeNotification->Type = "Notifications:User:Welcome";
            // $welcomeNotification->Data = new \stdClass();
                       
            $notificationDoc->Notifications = [$welcomeNotification];
            
            $this->em->getRepository(self::PHYSICAL_ENTITY)->save($notificationDoc);
        }
        
        return $notificationDoc;
    }

    private function addIdentity($physicalEntity, $identity) {
        if (!isset($physicalEntity->Identities)) {
            $physicalEntity->Identities = new \stdClass();
        }

        foreach ($identity as $key => $value) {
            $physicalEntity->Identities->$key = $value;
            $physicalEntity->Identities->$key->Verified = false;
        }
    }

    private function getProviderByKey(string $key) {
        foreach ($this->identityProviders as $identityProvider) {
            if ($identityProvider->getProviderKey() == $key) {
                return $identityProvider;
            }
        }
    }

    private function getProvider($identity) {
        if (!is_object($identity)) {
            throw new ValidationException(self::IDENTITY_NOT_VALID_ERROR, "Identity is not an object");
        }

        foreach ($this->identityProviders as $identityProvider) {
            if ($identityProvider->supports($identity)) {
                return $identityProvider;
            }
        }

        throw new ValidationException(self::IDENTITY_NOT_VALID_ERROR, "No identity provider found");
    }

    private function _reduceDocument($doc) {
        $ret = new \stdClass();
        $ret->_id = $doc->_id;
        $ret->CreationDate = $doc->CreationDate;
        $ret->UpdateDate = $doc->UpdateDate;

        $ret->FirstName = $doc->FirstName;
        $ret->LastName = $doc->LastName;
        $ret->FiscalCode = $doc->FiscalCode;
        $ret->Contacts = $doc->Contacts;

        if (isset($doc->UserDB) && isset($doc->UserPassword)) {
            $ret->UserDB = $doc->UserDB;
            $ret->UserPassword = $doc->UserPassword;
        }

        if (isset($doc->Roles)) {
            $ret->Roles = $doc->Roles;
        }

        if (isset($doc->Devices)) {
            $ret->Devices = $doc->Devices;
        }

        return $ret;
    }

    public function loginUser(\stdClass $identity, bool $createUserDB = false) {
        $identityProvider = $this->getProvider($identity);

        $providerKey = $identityProvider->getProviderKey();

        // normalize the identity
        $identityProvider->normalize($identity);

        // look for the physical entity with the provided identity
        $userByIdentity = $this->em->getRepository(self::PHYSICAL_ENTITY)->findOneByIdentity($providerKey, $identity->$providerKey->LoginKey);

        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }
        
        // verifica correttezza credenziali
        $identityProvider->checkCredentials($identity, $userByIdentity);

        // aggiungiamo la rappresentanza verso se stesso, se necessario
        $this->addUserLegalEntity($userByIdentity);

        $this->em->getRepository(self::PHYSICAL_ENTITY)->save($userByIdentity);

        // crea o aggiorna il db dell'utente
        if ($createUserDB) {
            $this->createOrRefreshUserDB($userByIdentity);
        }
        
        // proviamo ad aggiungere il documento per le notifiche, se necessario
        try {
            $this->addNotificationsDoc($userByIdentity);
        } catch (\Exception $ex) {
            // eventuali errori non bloccanti
        }        

        return $this->_reduceDocument($userByIdentity);
    }

    public function forgottenCredentials(string $providerKey, string $email) {
        // look for the physical entity with the provided email
        $userByEmail = $this->em->getRepository(self::PHYSICAL_ENTITY)->findOneByEmail(strtolower($email));

        if (!$userByEmail) {
            throw new ValidationException(self::USER_NOT_FOUND, "User with the given email not found");
        }

        $identityProvider = $this->getProviderByKey($providerKey);

        $result = $identityProvider->forgottenCredentials($userByEmail);

        $this->em->getRepository(self::PHYSICAL_ENTITY)->save($userByEmail);

        $this->sendForgottenCredentialsEmail($userByEmail, $result);

        return $result;
    }

    public function changeCredentials(string $providerKey, string $loginKey, \stdClass $changeCredentials) {
        $identityProvider = $this->getProviderByKey($providerKey);

        // look for the physical entity with the provided identity
        $userByIdentity = $this->em->getRepository(self::PHYSICAL_ENTITY)->findOneByIdentity($providerKey, strtolower($loginKey));

        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }

        $identityProvider->changeCredentials($userByIdentity, $changeCredentials);

        $this->em->getRepository(self::PHYSICAL_ENTITY)->save($userByIdentity);

        return $this->_reduceDocument($userByIdentity);
    }

    private function sendForgottenCredentialsEmail(\stdClass $physicalEntity, $resultForgottenCredentials) {
        $this->mailer->addAddress($physicalEntity->Contacts->Email->Value, $this->getFullName($physicalEntity));

        $this->translator->setLocale(isset($physicalEntity->Preferences->Language) ? $physicalEntity->Preferences->Language : "it");
        
        $this->mailer->isHTML();
        $this->mailer->Subject = $this->translator->trans('forgotten_credential_subject');
        
        $this->mailer->Body = $this->templating->render("forgottenCredentials.html.twig", [
            'translator' => $this->translator, 
            'password' => $resultForgottenCredentials->PlainPassword,
            'username' => $resultForgottenCredentials->Username
                ]);        

        return $this->mailer->send();
    }

    private function sendRegistrationEmail(\stdClass $physicalEntity) {
        $this->mailer->addAddress($physicalEntity->Contacts->Email->Value, $this->getFullName($physicalEntity));

        $this->translator->setLocale(isset($physicalEntity->Preferences->Language) ? $physicalEntity->Preferences->Language : "it");

        $this->mailer->isHTML();
        $this->mailer->Subject = $this->translator->trans('registration_subject');
        $this->mailer->Body = $this->templating->render("registration.html.twig", [
            'translator' => $this->translator, 
            'verificationCode' => $physicalEntity->Contacts->Email->VerificationCode,
            'email' => $physicalEntity->Contacts->Email->Value,
            'fullName' => $this->getFullName($physicalEntity)    
                ]);

        return $this->mailer->send();
    }

    private function getFullName(\stdClass $physicalEntity) {
        return $physicalEntity->FirstName . " " . $physicalEntity->LastName;
    }

    public function confirmRegistration(string $providerKey, string $loginKey, string $verificationCode, bool $createUserDB = false) {
        $identityProvider = $this->getProviderByKey($providerKey);

        // look for the physical entity with the provided identity
        $userByIdentity = $this->em->getRepository(self::PHYSICAL_ENTITY)->findOneByIdentity($providerKey, strtolower($loginKey));

        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }

        $identityProvider->confirmRegistration($userByIdentity, $verificationCode);

        $this->em->getRepository(self::PHYSICAL_ENTITY)->save($userByIdentity);

        // aggiunta db, se necessario
        if ($createUserDB) {
            try {
                $this->createOrRefreshUserDB($userByIdentity);
            } catch (\Exception $ex) {
                // eventuali errori non bloccanti
            }
        }        
        
        return $this->_reduceDocument($userByIdentity);
    }
    
    private function addUserLegalEntity(\stdClass $physicalEntity) {
        // verifichiamo se esiste la legal entity col codice fiscale della physical entity
        $legalEntity = $this->em->getRepository(self::LEGAL_ENTITY)->findOneByFiscalCode($physicalEntity->FiscalCode);
        
        $checkForRole = true;
        if (is_null($legalEntity)) {
            // definiamo la nuova legal entity
            $legalEntity = new \stdClass();
            $legalEntity->_id = $this->em->getRepository(self::LEGAL_ENTITY)->generateId($physicalEntity->FiscalCode);
            $legalEntity->Type = "LegalEntity";
            $legalEntity->FiscalCode = $physicalEntity->FiscalCode;
            $legalEntity->Name = $this->getFullName($physicalEntity);
            $legalEntity->Status = "LegalEntity:Active";
            
            $this->em->getRepository(self::LEGAL_ENTITY)->save($legalEntity);
            $checkForRole = false;
        }     
        
        // verifichiamo se ha già la rappresentanza
        if (isset($physicalEntity->Roles) && $checkForRole) {
            foreach ($physicalEntity->Roles as $existingRole) {
                if ($existingRole->legalEntity == $legalEntity->_id) {
                    return;
                }
            }
        } 
        
        if (!isset($physicalEntity->Roles)) {
            $physicalEntity->Roles = [];
        }
        
        $role = new \stdClass();
        $role->Level = "legalEntity";
        $role->legalEntity = $legalEntity->_id;
        $role->Role = "Role:LegalRepresentative";

        $physicalEntity->Roles[] = $role;
    }
    
    private function createOrRefreshUserDB($user) {
        // crea il db dell'utente, se non esiste
        if (!$this->createDatabaseUserInfo($user)) {

            // aggiorna i documenti di replica
            if($this->createOrUpdateReplicators($user)) {
                $this->em->getRepository(self::PHYSICAL_ENTITY)->save($user);
            }

            // aggiorna i roles nel doc in _users
            $this->updateDatabaseUser($user);
        }        
    }

	private function createDatabaseUserInfo($phisicalEntity){
        
		$userID = "org.couchdb.user:".$phisicalEntity->_id;
        $now = new \DateTime();
        
        try {
            $user = $this->userCouchDBClient->getDoc($userID);
            $user->update_date = $now->format(\DateTime::ATOM);
            $this->userCouchDBClient->storeDoc($user);
            return false;
        } catch (\Exception $e) {
            $userPassword = isset($phisicalEntity->UserPassword) ? $phisicalEntity->UserPassword : uniqid();
            
            // documento in _users non trovato ... posso quindi creare la grant
            $user = new \stdClass();
            $user->_id = $userID;
            $user->name = $phisicalEntity->_id;
            $user->password = $userPassword;
            $user->type = 'user';
            $user->roles = $this->calculateDatabaseUserRoles($phisicalEntity);
            $user->creation_date = $now->format(\DateTime::ATOM);
            $user->update_date = $now->format(\DateTime::ATOM);
            $this->userCouchDBClient->storeDoc($user);
        
            $userCouchDBName = 'userdb-'.$this->_str2hex($phisicalEntity->_id);

            $phisicalEntity->UserDB = new \stdClass();
            $phisicalEntity->UserDB->Name = $userCouchDBName;
            $phisicalEntity->UserDB->Username = $phisicalEntity->_id;
            $phisicalEntity->UserDB->Protocol = $this->userCouchDBProtocol;
            $phisicalEntity->UserDB->Host = $this->userCouchDBHost;
            $phisicalEntity->UserDB->Port = $this->userCouchDBPort;
            $phisicalEntity->UserPassword = $userPassword;
            
            $this->createOrUpdateReplicators($phisicalEntity);
            
            $this->em->getRepository(self::PHYSICAL_ENTITY)->save($phisicalEntity);
            
            return true;
        }
	}  
    
	private function _str2hex($func_string) {
		$func_retVal = '';
		$func_length = strlen($func_string);
		for($func_index = 0; $func_index < $func_length; ++$func_index) {
            $func_retVal .= ((($c = dechex(ord($func_string{$func_index}))) && strlen($c) & 2) ? $c : "0{$c}");
        }

		return strtolower($func_retVal);
	}
    
    private function calculateDatabaseUserRoles($phisicalEntity) {
		$UserRoles = array("SnapSafe");
        
        if(!isset($phisicalEntity->Roles)){
			return $UserRoles;
		}
        
		foreach ($phisicalEntity->Roles as $role) {
            if (isset($role->Division)) {
                $UserRoles[] = $role->Division;
            } else if(isset($role->legalEntity)) {
                $UserRoles[] = $role->legalEntity;
            }
        } 
        
        return $UserRoles;
    }
    
	private function createOrUpdateReplicators($physicalEntity){
		
        $updatePhysicalEntity = false;
		$userCouchDBName = $physicalEntity->UserDB->Name;
        $replicatorDownwardId = "downward:".$userCouchDBName;
        
        if (!isset($physicalEntity->UserDB->Replicators)) {
            $physicalEntity->UserDB->Replicators = new \stdClass();
            $updatePhysicalEntity = true;
        }
        
        // authorizations
        $appAuthorization = "Basic ".base64_encode($this->appCouchDBUsername.":".$this->appCouchDBPassword);
		
		// downward
        $createReplicatorDownward = true;
        if (isset($physicalEntity->UserDB->Replicators->Downward)) {
            try {
                $replicatorDownward =  $this->replicationCouchDBClient->getDoc($physicalEntity->UserDB->Replicators->Downward);
                $createReplicatorDownward = false;
                
                $newDocumentSelector = $this->createDocumentSelector($physicalEntity);
                $oldDocumentSelector = $replicatorDownward->selector;
                if($newDocumentSelector != $oldDocumentSelector){
                    $replicatorDownward->selector = $newDocumentSelector;
                }  
                
                $now = new \DateTime();
                $replicatorDownward->update_date = $now->format(\DateTime::ATOM);                
                
                $this->replicationCouchDBClient->storeDoc($replicatorDownward);          
            } catch (\Exception $ex) {

            }
        }
        
        if ($createReplicatorDownward) {
            $sourceUrl = $this->appCouchDBHost."/".$this->appCouchDBName;
            $targetUrl = $this->appCouchDBHost."/".$userCouchDBName;
            $replicatorDownward = new \stdClass();
            $replicatorDownward->_id = $replicatorDownwardId;
            
			$replicatorDownward->source = new \stdClass();
			$replicatorDownward->source->url = $sourceUrl;
			$replicatorDownward->source->headers = new \stdClass();
			$replicatorDownward->source->headers->Authorization = $appAuthorization;
            
			$replicatorDownward->target = new \stdClass();
			$replicatorDownward->target->url = $targetUrl;
			$replicatorDownward->target->headers = new \stdClass();
			$replicatorDownward->target->headers->Authorization = $appAuthorization;

            $replicatorDownward->user_ctx = new \stdClass();
            $replicatorDownward->user_ctx->name = $physicalEntity->_id;
            $replicatorDownward->user_ctx->roles = [];	            

            $replicatorDownward->create_target = FALSE;
            $replicatorDownward->owner = $physicalEntity->_id;
            $replicatorDownward->continuous = TRUE;
            $replicatorDownward->selector = $this->createDocumentSelector($physicalEntity);
            
            $now = new \DateTime();
            $replicatorDownward->creation_date = $now->format(\DateTime::ATOM);
            $replicatorDownward->update_date = $now->format(\DateTime::ATOM);
            
            $this->replicationCouchDBClient->storeDoc($replicatorDownward);
            
            $physicalEntity->UserDB->Replicators->Downward = $replicatorDownwardId;
            $updatePhysicalEntity = true;
        }  
        
        return $updatePhysicalEntity;
	}   
    
	private function createDocumentSelector($phisicalEntity){
		$selector = new \stdClass();
		$selector->{'$or'} = array();
		
        // documenti con indici?
		$indexesSelector = new \stdClass();
		$regexRule = new \stdClass();
		$rule = "\$regex";
		$regexRule->$rule = "^_design/userdb_.*";
		$indexesSelector->_id = $regexRule; 
        $selector->{'$or'}[] = $indexesSelector;
        
        // documenti con funzioni di validazione
		$validationSelector = new \stdClass();
		$validationRegexRule = new \stdClass();
		$validationRule = "\$regex";
		$validationRegexRule->$validationRule = "^_design/validation_.*";
		$validationSelector->_id = $validationRegexRule;
        $selector->{'$or'}[] = $validationSelector;
      
        // own document
        $ownDocumentSelector =  new \stdClass();
		$ownDocumentSelector->_id = $phisicalEntity->_id;
        $selector->{'$or'}[] = $ownDocumentSelector;
        
        // notifications document
		$notificationSelector = new \stdClass();
		$notificationRegexRule = new \stdClass();
		$notificationRule = "\$regex";
		$notificationRegexRule->$notificationRule = "^".$this->em->getRepository(self::PHYSICAL_ENTITY)->getNotificationsDocIdPrefix($phisicalEntity->FiscalCode)."*";
		$notificationSelector->_id = $notificationRegexRule;
        $selector->{'$or'}[] = $notificationSelector;
        
        // language document
        $languageSelector =  new \stdClass();
		$languageSelector->Type = "Language";
        $selector->{'$or'}[] = $languageSelector;        
        
        // roles
		if(!isset($phisicalEntity->Roles)){
			return $selector;
		}
		
		foreach ($phisicalEntity->Roles as $role) {
			$level = $role->Level;
            $subject = null;
            
            // legalEntity
            if (isset($role->legalEntity)) {
                $legalEntitySelector = new \stdClass();
                $legalEntitySelector->_id = $role->legalEntity;
                $selector->{'$or'}[] = $legalEntitySelector;
				
				$legalEntitySelector2 = new \stdClass();
                $legalEntitySelector2->legalEntityId = $role->legalEntity;
				$legalEntitySelector2->Type = "Division";
                $selector->{'$or'}[] = $legalEntitySelector2;
                
                if ($level == "legalEntity") {
                    $subject = $role->legalEntity;
                }
            }
            
            // Division
            if ($level == "Division" && isset($role->Division)) {
                $divisionSelector = new \stdClass();
                $divisionSelector->_id = $role->Division;
                $selector->{'$or'}[] = $divisionSelector;
                
                $subject = $role->Division;
            }
            
            // Role
            if (!is_null($subject)) {
                $roleSelector = new \stdClass();
                $roleSelector->{"Owner.$level"} = $subject;
                $selector->{'$or'}[] = $roleSelector; 
            }
		}
				
		return $selector;
	}

    private function updateDatabaseUser($physicalEntity) {
		$userID = "org.couchdb.user:".$physicalEntity->_id;   
        
        $user = $this->userCouchDBClient->getDoc($userID);
        
		$newRoles = $this->calculateDatabaseUserRoles($physicalEntity);
        
        $rolesDiff1 = array_diff($user->roles, $newRoles);
        $rolesDiff2 = array_diff($newRoles, $user->roles);
        if (count($rolesDiff1) == 0 && count($rolesDiff2) == 0) {
            return;
        }
        
        $user->roles = $newRoles;
        
		$this->userCouchDBClient->storeDoc($user);      
    }    
    
    public function alive(string $userId) {
        $physicalEntity = $this->em->getRepository(self::PHYSICAL_ENTITY)->getById($userId);
        
        // crea o aggiorna il db dell'utente
        $this->createOrRefreshUserDB($physicalEntity);
        
        // proviamo ad aggiungere il documento per le notifiche, se necessario
        try {
            $this->addNotificationsDoc($physicalEntity);
        } catch (\Exception $ex) {
            // eventuali errori non bloccanti
        }        

        // calcola le info da ritornare
        return $this->_reduceDocument($physicalEntity);
    }    
    
    public function addDevice(string $userId, string $deviceType, string $deviceId) {
             
		if(strlen($deviceType) < 1){
			throw new ValidationException(self::DEVICE_TYPE_NOT_VALID, "Device type cant't be an empty string");
		} 
        
		if(strlen($deviceId) < 1){
			throw new ValidationException(self::DEVICE_ID_NOT_VALID, "Device id cant't be an empty string");
		}
	
		$physicalEntity = $this->em->getRepository(self::PHYSICAL_ENTITY)->getById($userId);
		
		// check existent $deviceId
		$deviceIdFound = FALSE;
        if (!isset($physicalEntity->Devices)) {
            $physicalEntity->Devices = [];
        }
        
        foreach ($physicalEntity->Devices as $device) {
            if($device->Id === $deviceId){
                $deviceIdFound = TRUE;
                break;
            }
        }
		
		//Add Device only if device ID is not present
		if($deviceIdFound === FALSE){
			$Now = new \DateTime();
			$device = new \stdClass();
			$device->Id = $deviceId;
			$device->Type = $deviceType;
			$device->CreationDate = $Now->format(\DateTime::ATOM);

			$physicalEntity->Devices[] = $device;

			$this->em->getRepository(self::PHYSICAL_ENTITY)->save($physicalEntity);
		}
		
		return $this->_reduceDocument($physicalEntity);        
    }
	
	public function removeDevice(string $userId, string $deviceId){
		if(strlen($deviceId) < 1){
			throw new ValidationException(self::DEVICE_ID_NOT_VALID, "Device id cant't be an empty string");
		}
	
		$physicalEntity = $this->em->getRepository(self::PHYSICAL_ENTITY)->getById($userId);
		
		if(is_array($physicalEntity->Devices) && count($physicalEntity->Devices) != 0){
			foreach ($physicalEntity->Devices as $key => $device){
				if($device->Id == $deviceId){
					unset($physicalEntity->Devices[$key]);
					$physicalEntity->Devices = array_values($physicalEntity->Devices);
					$this->em->getRepository(self::PHYSICAL_ENTITY)->save($physicalEntity);
					break;
				}
			}
		}
		
		return $this->_reduceDocument($physicalEntity);
	}
}
