<?php

namespace Schema31\SafeHubLib\Repository;

use Schema31\CouchDBClient\Service\CouchDBClient;
use Schema31\CouchDBClient\Domain\ViewRequest;
use Schema31\SafeHubLib\Exception\UnexpectedException;

/**
 * Description of BaseRepository
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class BaseRepository {
    protected $couchDBClient;
    
    const COUCHDB_GENERIC_ERROR = '0002';
    const NOT_UNIQUE_RESULT_ERROR = '0003';
	const NOT_FOUND_ERROR = '0004';
        
    public function __construct(CouchDBClient $couchDBClient) {
        $this->couchDBClient = $couchDBClient;
    }
    
    public function generateUUID() {
        try {
            $uuids = $this->couchDBClient->getUuids(1);
            return $uuids[0];
        } catch (\Exception $e) {
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "Error retrieving UUIDs");
        }        
    }
    
    protected function beforeSave(\stdClass $object) {
        $now = (new \DateTime())->format(\DateTime::ATOM);
        $object->UpdateDate = $now;
        
        if (!isset($object->CreationDate)) {
            $object->CreationDate = $now;
        }    
    } 

    public function save(\stdClass $object) {
        try {
            $this->beforeSave($object);
            $this->couchDBClient->storeDoc($object);
        } catch (\Exception $e) {
            $details = [];
                    
            if (isset($object->Type)) {
                $details['Type'] = $object->Type;
            } 
            
            if (isset($object->_id)) {
                $details['_id'] = $object->_id;
            }             
            
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "Error saving document: ". json_encode($details));
        } 
    } 
    
    protected function findOneByViewRequest(ViewRequest $viewRequest, array $details = []) {
        $details[] = get_class();
        try {
            $viewResponse = $this->couchDBClient->getView($viewRequest);
        } catch (\Exception $e) {
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "findOneByViewRequest error: ". json_encode($details));
        }
        
        $results = $viewResponse->getDocs();
        
        if (count($results) > 1) {
            throw new UnexpectedException(self::NOT_UNIQUE_RESULT_ERROR, "findOneByViewRequest error, more than one result found: ". json_encode($details));
        } 
        
        return count($results) == 0 ? null : $results[0];
    }
    
    public function getById($docId) {
		try{
			return $this->couchDBClient->getDoc($docId);
		} catch (\Schema31\CouchDBClient\Exception\NotFoundException $ex) {
			throw new \Schema31\SafeHubLib\Exception\NotFoundException(self::NOT_FOUND_ERROR, "Document with ID $docId not found");
		} catch (\Exception $ex) {
			throw new \Schema31\SafeHubLib\Exception\UnexpectedException(self::COUCHDB_GENERIC_ERROR, "CouchDB generic error");
		}
    }
}
