<?php

namespace Schema31\SafeHubLib\IdentityProvider;

use Schema31\SafeHubLib\Exception\ValidationException;

/**
 * Description of OneProfileIdentityProvider
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class OneProfileIdentityProvider implements IdentityProviderInterface {
    const PASSWORD_NOT_PRESENT_ERROR = "U-0012";
    const LOGIN_PASSWORD_NOT_PRESENT_ERROR = "U-0004";
    const LOGIN_PASSWORD_NOT_VALID_ERROR = "U-0001";
    const LOGIN_IDENTITY_NOT_FOUND = "U-0013";
    
    const SALT_LENGTH = 10;
    
    public function supports(\stdClass $identity) {
        return isset($identity->OneProfile);
    }

    public function normalize(\stdClass $identity) {
        if (isset($identity->OneProfile->Username)) {
            $identity->OneProfile->Username = strtolower($identity->OneProfile->Username);
            $identity->OneProfile->LoginKey = $identity->OneProfile->Username;
        }		        
    }

    public function getProviderKey() {
        return "OneProfile";
    }   

    public function finalize(\stdClass $identity) {
        $this->encryptSaltPassword($identity->OneProfile, $identity->OneProfile->Password);  
    }
    
    private function encryptSaltPassword(\stdClass $host, $plainPassword) {
        $salt = substr(md5(uniqid(rand(), true)), 0, self::SALT_LENGTH);
        $passwordCripted = sha1($plainPassword . $salt); 
        
        $host->Salt = $salt;
        $host->Password = $passwordCripted;           
    }

    public function validate(\stdClass $identity) {
    	if(!isset($identity->OneProfile->Password)){
            throw new ValidationException(self::PASSWORD_NOT_PRESENT_ERROR, "Safe identity password not provided");
        }
    }

    public function isEmailVerified() {
        return false;
    }

    public function checkCredentials(\stdClass $identity, \stdClass $physicalEntity) {
        // check if password was provided
        if (!isset($identity->OneProfile->Password)) {
            throw new ValidationException(self::LOGIN_PASSWORD_NOT_PRESENT_ERROR, "OneProfile identity password not provided");
        }

        if(isset($physicalEntity->Identities) && isset($physicalEntity->Identities->OneProfile)){
            if (!$this->_check_password($identity->OneProfile->Password, $physicalEntity)) {
                throw new ValidationException(self::LOGIN_PASSWORD_NOT_VALID_ERROR, "OneProfile identity password not valid");
            }
            
            return true;
        }

        throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "OneProfile identity not found");
    }

	private function _check_password($inputPassword, $physicalEntity) {       
        $host = $physicalEntity->Identities->OneProfile;
        
        // password normale
        if ($host->Password == sha1($inputPassword . $host->Salt)) {
            return true;
        }
        
        // password per recupero credenziali
        if (isset($host->ForgottenCredentials) && $host->ForgottenCredentials->Password == sha1($inputPassword . $host->ForgottenCredentials->Salt)) {
            $host->Password = $host->ForgottenCredentials->Password;
            $host->Salt = $host->ForgottenCredentials->Salt;
            unset($host->ForgottenCredentials);
            return true;
        }	
        
        return false;
	}

    public function forgottenCredentials(\stdClass $physicalEntity) {
        $physicalEntity->Identities->OneProfile->ForgottenCredentials = new \stdClass();
        
        $plainPassword = $this->generatePassword(8);
        $this->encryptSaltPassword($physicalEntity->Identities->OneProfile->ForgottenCredentials, $plainPassword);
        
        $forgottenCredentialsResult = new \stdClass();
        $forgottenCredentialsResult->PlainPassword = $plainPassword;
        
        return $forgottenCredentialsResult;
    }

    public function generatePassword($chars) 
    {
      $data = '1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcefghijklmnopqrstuvwxyz';
      return substr(str_shuffle($data), 0, $chars);
    }    
}
