<?php

namespace Schema31\SafeHubLib\Service;

use Schema31\SafeHubLib\IdentityProvider\SafeIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\GoogleIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\FacebookIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\OneProfileIdentityProvider;

use Schema31\SafeHubLib\Repository\PhysicalEntityRepository;

use Schema31\SafeHubLib\Exception\ValidationException;

/**
 * Description of UserService
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class UserService {
    
    const IDENTITY_NOT_VALID_ERROR = "U-0002";
    const IDENTITY_ALREADY_PRESENT_ERROR = "U-0011";
   
    const DOCUMENT_TEMPLATE = 1;
    
    private $identityProviders;
    private $physicalEntityRepository;
    
    public function __construct(PhysicalEntityRepository $physicalEntityRepository) {
        $this->identityProviders = [
            new SafeIdentityProvider(),
            new GoogleIdentityProvider(),
            new FacebookIdentityProvider(),
            new OneProfileIdentityProvider()
        ];
        
        $this->physicalEntityRepository = $physicalEntityRepository;
    }
    
    /**
     * Create a basic user document
     * 
     * @param string $fiscalCode
     * @return \stdClass
     */
    public function createUserDocument(string $fiscalCode) {
        $physicalEntity = new \stdClass();
        $physicalEntity->_id = $this->physicalEntityRepository->generateId($fiscalCode);
        $physicalEntity->Type = "PhysicalEntity";
        $physicalEntity->DocumentTemplate = self::DOCUMENT_TEMPLATE;
        $physicalEntity->FiscalCode = $fiscalCode;   
        
        return $physicalEntity;
    }
    
    public function createUser(string $firstName, string $lastName, string $fiscalCode, string $email, \stdClass $identity, bool $createUserDB = false) {
		$identityProvider = $this->getProvider($identity);

        // validate the identity
        $identityProvider->validate($identity);        
        
        // normalize the identity
        $identityProvider->normalize($identity);
        
        $providerKey = $identityProvider->getProviderKey();
		
        // look for existing physical entities with the provided identity
        $userByIdentity = $this->physicalEntityRepository->findOneByIdentity($providerKey, $identity->$providerKey->LoginKey);
        
        if($userByIdentity){
            throw new ValidationException(self::IDENTITY_ALREADY_PRESENT_ERROR, "Provided identity already present");
        }
		
        // look for existing physical entities with the provided fiscal code
		$physicalEntity = $this->physicalEntityRepository->findOneByFiscalCode($fiscalCode);

        // if exists
        if ($physicalEntity) {
            // and is already a user and has the provided identity
            if (isset($physicalEntity->Identities->$providerKey)) {
                // is really needed this check?
                throw new ValidationException(self::IDENTITY_ALREADY_PRESENT_ERROR, "Provided identity already present");
            }
            
            $identity->$providerKey->VerificationCode = substr(bin2hex(random_bytes(5)),0,4);          
        } else {
            // physical entity creation
            $physicalEntity = $this->createUserDocument($fiscalCode);
            $physicalEntity->FirstName = $firstName;
            $physicalEntity->LastName = $lastName;
            $physicalEntity->UserDBRequested = $createUserDB;
            
			$contacts = new \stdClass();
			$contacts->Email = new \stdClass();
			$contacts->Email->Value = $email;
			$contacts->Email->Verified = $identityProvider->isEmailVerified();
			if(!$contacts->Email->Verified){
				$contacts->Email->VerificationCode = substr(bin2hex(random_bytes(5)),0,4);
			}
            
			$physicalEntity->Contacts = $contacts;         
        }
        
        $identityProvider->finalize($identity);
        
        // add identity
		$this->addIdentity($physicalEntity, $identity);
        
        // aggiungiamo la rappresentanza verso se stesso
        // $this->addUserLegalEntity($PhisicalEntity);
        
        $this->beforeSave($physicalEntity);
        $this->physicalEntityRepository->save($physicalEntity);
        
        // aggiunta db, se necessario
        if ($createUserDB) {
            // $this->createDatabaseUserInfo($physicalEntity);
        }
		
		return $this->_reduceDocument($physicalEntity);
    } 
    
    private function addIdentity($physicalEntity, $identity) {     
        if (!isset($physicalEntity->Identities)) {
            $physicalEntity->Identities = new \stdClass();
        }
        
        foreach ($identity as $key => $value) {
            $physicalEntity->Identities->$key = $value;
            $physicalEntity->Identities->$key->Verified = false;
        }
    }    
    
    private function getProvider($identity) {
        if (!is_object($identity)) {
            throw new ValidationException(self::IDENTITY_NOT_VALID_ERROR, "Identity is not an object");
        }
        
        foreach ($this->identityProviders as $identityProvider) {
            if ($identityProvider->supports($identity)) {
                return $identityProvider;
            }
        }
        
        throw new ValidationException(self::IDENTITY_NOT_VALID_ERROR, "No identity provider found");
    }
    
	private function _reduceDocument($doc) {
    	$ret = new \stdClass();
		$ret->_id = $doc->_id;
        $ret->CreationDate = $doc->CreationDate;
        $ret->UpdateDate = $doc->UpdateDate;
        
		$ret->FirstName = $doc->FirstName;
		$ret->LastName = $doc->LastName;
		$ret->FiscalCode = $doc->FiscalCode;       
		$ret->Contacts = $doc->Contacts;
        
        if (isset($doc->UserDB) && isset($doc->UserPassword)) {
            $ret->UserDB = $doc->UserDB;
            $ret->UserPassword = $doc->UserPassword;
        }
        
        if (isset($doc->Roles)) {
            $ret->Roles = $doc->Roles;
        }
		
		if (isset($doc->Devices)) {
            $ret->Devices = $doc->Devices;
        }
        
		return $ret;
    }    
    
    public function beforeSave($physicalEntity) {
        $now = (new \DateTime())->format(\DateTime::ATOM);
        $physicalEntity->UpdateDate = $now;
        
        if (!isset($physicalEntity->CreationDate)) {
            $physicalEntity->CreationDate = $now;
        }    
    }

	public function loginUser(\stdClass $identity) {
        $identityProvider = $this->getProvider($identity); 
		
        $providerKey = $identityProvider->getProviderKey();
        
        // normalize the identity
        $identityProvider->normalize($identity);        
        
        // look for the physical entity with the provided identity
        $userByIdentity = $this->physicalEntityRepository->findOneByIdentity($providerKey, $identity->$providerKey->LoginKey);
		
        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }
        
        // se esiste, ritorno il documento parziale al chiamante

        $identityProvider->checkCredentials($identity, $userByIdentity);

        // aggiungiamo la rappresentanza verso se stesso, se necessario
        // $this->addUserLegalEntity($userByIdentity);  

        // crea o aggiorna il db dell'utente
        if ($userByIdentity->UserDBRequested) {
            // $this->createOrRefreshUserDB($userByIdentity);
        }

        return $this->_reduceDocument($userByIdentity);
    }
    
}
