<?php

namespace Schema31\SafeHubLib\IdentityProvider;

use Schema31\SafeHubLib\Exception\ValidationException;

/**
 * Description of OneProfileIdentityProvider
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class OneProfileIdentityProvider implements IdentityProviderInterface {
    const PASSWORD_NOT_PRESENT_ERROR = "U-0012";
    const LOGIN_PASSWORD_NOT_PRESENT_ERROR = "U-0004";
    const LOGIN_PASSWORD_NOT_VALID_ERROR = "U-0001";
    const LOGIN_IDENTITY_NOT_FOUND = "U-0013";
    
    public function supports(\stdClass $identity) {
        return isset($identity->OneProfile);
    }

    public function normalize(\stdClass $identity) {
        if (isset($identity->OneProfile->Username)) {
            $identity->OneProfile->Username = strtolower($identity->OneProfile->Username);
            $identity->OneProfile->LoginKey = $identity->OneProfile->Username;
        }		        
    }

    public function getProviderKey() {
        return "OneProfile";
    }
    
    public function salt() {
        $salt_length = 10;
		return substr(md5(uniqid(rand(), true)), 0, $salt_length);
	}    

    public function finalize(\stdClass $identity) {
        $password = $identity->OneProfile->Password;
        $passwordCripted = $this->_crypt_password($password);
        $identity->OneProfile->Password = $passwordCripted;        
    }

	private function _crypt_password($input) {        
		return sha1($input . $this->salt());
	}

    public function validate(\stdClass $identity) {
    	if(!isset($identity->OneProfile->Password)){
            throw new ValidationException(self::PASSWORD_NOT_PRESENT_ERROR, "Safe identity password not provided");
        }
    }

    public function isEmailVerified() {
        return false;
    }

    public function checkCredentials(\stdClass $identity, \stdClass $physicalEntity) {
        // check if password was provided
        if (!isset($identity->OneProfile->Password)) {
            throw new ValidationException(self::LOGIN_PASSWORD_NOT_PRESENT_ERROR, "OneProfile identity password not provided");
        }

        if(isset($physicalEntity->Identities) && isset($physicalEntity->Identities->OneProfile)){
            if (!$this->_check_password($identity->OneProfile->Password, $physicalEntity)) {
                throw new ValidationException(self::LOGIN_PASSWORD_NOT_VALID_ERROR, "OneProfile identity password not valid");
            }
            
            return true;
        }

        throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "OneProfile identity not found");
    }

	private function _check_password($inputPassword, $physicalEntity) {
		return $physicalEntity->Identities->OneProfile->Password == sha1($inputPassword . $physicalEntity->Identities->OneProfile->Salt);
	}     
}
