<?php

namespace Schema31\SafeHubLib\IdentityProvider;

use Schema31\SafeHubLib\Exception\ValidationException;

/**
 * Description of SafeIdentityProvider
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class SafeIdentityProvider implements IdentityProviderInterface {
    const PASSWORD_NOT_PRESENT_ERROR = "U-0012";
    const LOGIN_PASSWORD_NOT_PRESENT_ERROR = "U-0004";
    const LOGIN_PASSWORD_NOT_VALID_ERROR = "U-0001";
    const LOGIN_IDENTITY_NOT_FOUND = "U-0013";
    
    public function supports(\stdClass $identity) {
        return isset($identity->Safe);
    }

    public function normalize(\stdClass $identity) {
        if (isset($identity->Safe->Username)) {
            $identity->Safe->Username = strtolower($identity->Safe->Username);
            $identity->Safe->LoginKey = $identity->Safe->Username;
        }		        
    }

    public function getProviderKey() {
        return "Safe";
    }

    public function finalize(\stdClass $identity) {
        $password = $identity->Safe->Password;
        $passwordCripted = $this->_crypt_password($password);
        $identity->Safe->Password = $passwordCripted;        
    }

	private function _crypt_password($input) {
		$crypt_options = [
		  'cost' => 10
		];
        
		return password_hash($input, PASSWORD_BCRYPT, $crypt_options);
	}

    public function validate(\stdClass $identity) {
    	if(!isset($identity->Safe->Password)){
            throw new ValidationException(self::PASSWORD_NOT_PRESENT_ERROR, "Safe identity password not provided");
        }
    }

    public function isEmailVerified() {
        return false;
    }

    public function checkCredentials(\stdClass $identity, \stdClass $physicalEntity) {
        // check if password was provided
        if (!isset($identity->Safe->Password)) {
            throw new ValidationException(self::LOGIN_PASSWORD_NOT_PRESENT_ERROR, "Safe identity password not provided");
        }

        if(isset($physicalEntity->Identities) && isset($physicalEntity->Identities->Safe)){
            if (!$this->_check_password($identity->Safe->Password, $physicalEntity->Identities->Safe->Password)) {
                throw new ValidationException(self::LOGIN_PASSWORD_NOT_VALID_ERROR, "Safe identity password not valid");
            }
            
            return true;
        }

        throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
    }

	private function _check_password($inputPassword, $hashedPassword) {
		return password_verify($inputPassword, $hashedPassword);
	}     
}
