<?php

namespace Schema31\SafeHubLib\Service;

use Schema31\SafeHubLib\IdentityProvider\SafeIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\GoogleIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\FacebookIdentityProvider;
use Schema31\SafeHubLib\IdentityProvider\OneProfileIdentityProvider;

use Schema31\SafeHubLib\Repository\PhysicalEntityRepository;

use Schema31\SafeHubLib\Exception\ValidationException;

/**
 * Description of UserService
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class UserService {
    
    const IDENTITY_NOT_VALID_ERROR = "U-0002";
    const IDENTITY_ALREADY_PRESENT_ERROR = "U-0011";
    const LOGIN_IDENTITY_NOT_FOUND = "U-0013";
    const USER_NOT_FOUND = "U-0010";
   
    const DOCUMENT_TEMPLATE = 1;
    
    private $identityProviders;
    private $physicalEntityRepository;
    
    public function __construct(PhysicalEntityRepository $physicalEntityRepository) {
        $this->identityProviders = [
            new SafeIdentityProvider(),
            new GoogleIdentityProvider(),
            new FacebookIdentityProvider(),
            new OneProfileIdentityProvider()
        ];
        
        $this->physicalEntityRepository = $physicalEntityRepository;
    }
    
    /**
     * Create a basic user document
     * 
     * @param string $fiscalCode
     * @return \stdClass
     */
    public function createUserDocument(string $fiscalCode) {
        $physicalEntity = new \stdClass();
        $physicalEntity->_id = $this->physicalEntityRepository->generateId($fiscalCode);
        $physicalEntity->Type = "PhysicalEntity";
        $physicalEntity->DocumentTemplate = self::DOCUMENT_TEMPLATE;
        $physicalEntity->FiscalCode = $fiscalCode;   
        
        return $physicalEntity;
    }
    
    public function createUser($mailer, string $firstName, string $lastName, string $fiscalCode, string $email, \stdClass $identity, string $language, bool $createUserDB = false) {
		$identityProvider = $this->getProvider($identity);

        // validate the identity
        $identityProvider->validate($identity);        
        
        // normalize the identity
        $identityProvider->normalize($identity);
        
        $providerKey = $identityProvider->getProviderKey();
		
        // look for existing physical entities with the provided identity
        $userByIdentity = $this->physicalEntityRepository->findOneByIdentity($providerKey, $identity->$providerKey->LoginKey);
        
        if($userByIdentity){
            throw new ValidationException(self::IDENTITY_ALREADY_PRESENT_ERROR, "Provided identity already present");
        }
		
        // look for existing physical entities with the provided fiscal code
		$physicalEntity = $this->physicalEntityRepository->findOneByFiscalCode($fiscalCode);

        // if exists
        if ($physicalEntity) {
            // and is already a user and has the provided identity
            if (isset($physicalEntity->Identities->$providerKey)) {
                // is really needed this check?
                throw new ValidationException(self::IDENTITY_ALREADY_PRESENT_ERROR, "Provided identity already present");
            }
            
            $identity->$providerKey->VerificationCode = substr(bin2hex(random_bytes(5)),0,4);          
        } else {
            // physical entity creation
            $physicalEntity = $this->createUserDocument($fiscalCode);
            $physicalEntity->FirstName = $firstName;
            $physicalEntity->LastName = $lastName;         
        }
        
        $physicalEntity->UserDBRequested = $createUserDB;
        
        $contacts = new \stdClass();
        $contacts->Email = new \stdClass();
        $contacts->Email->Value = $email;
        $contacts->Email->Verified = $identityProvider->isEmailVerified();
        if(!$contacts->Email->Verified){
            $contacts->Email->VerificationCode = substr(bin2hex(random_bytes(5)),0,4);
        }
        $physicalEntity->Contacts = $contacts; 

        $preferences = new \stdClass();
        $preferences->Language = $language;
        $physicalEntity->Preferences = $preferences;
        
        $identityProvider->finalize($identity);
        
        // add identity
		$this->addIdentity($physicalEntity, $identity);
        
        // aggiungiamo la rappresentanza verso se stesso
        // $this->addUserLegalEntity($PhisicalEntity);
        
        $this->physicalEntityRepository->save($physicalEntity);
        
        // aggiunta db, se necessario
        if ($createUserDB) {
            // $this->createDatabaseUserInfo($physicalEntity);
        }
        
        $this->sendRegistrationEmail($mailer, $physicalEntity);
		
		return $this->_reduceDocument($physicalEntity);
    } 
    
    private function addIdentity($physicalEntity, $identity) {     
        if (!isset($physicalEntity->Identities)) {
            $physicalEntity->Identities = new \stdClass();
        }
        
        foreach ($identity as $key => $value) {
            $physicalEntity->Identities->$key = $value;
            $physicalEntity->Identities->$key->Verified = false;
        }
    }

    private function getProviderByKey(string $key) {
        foreach ($this->identityProviders as $identityProvider) {
            if ($identityProvider->getProviderKey() == $key) {
                return $identityProvider;
            }
        }        
    }
    
    private function getProvider($identity) {
        if (!is_object($identity)) {
            throw new ValidationException(self::IDENTITY_NOT_VALID_ERROR, "Identity is not an object");
        }
        
        foreach ($this->identityProviders as $identityProvider) {
            if ($identityProvider->supports($identity)) {
                return $identityProvider;
            }
        }
        
        throw new ValidationException(self::IDENTITY_NOT_VALID_ERROR, "No identity provider found");
    }
    
	private function _reduceDocument($doc) {
    	$ret = new \stdClass();
		$ret->_id = $doc->_id;
        $ret->CreationDate = $doc->CreationDate;
        $ret->UpdateDate = $doc->UpdateDate;
        
		$ret->FirstName = $doc->FirstName;
		$ret->LastName = $doc->LastName;
		$ret->FiscalCode = $doc->FiscalCode;       
		$ret->Contacts = $doc->Contacts;
        
        if (isset($doc->UserDB) && isset($doc->UserPassword)) {
            $ret->UserDB = $doc->UserDB;
            $ret->UserPassword = $doc->UserPassword;
        }
        
        if (isset($doc->Roles)) {
            $ret->Roles = $doc->Roles;
        }
		
		if (isset($doc->Devices)) {
            $ret->Devices = $doc->Devices;
        }
        
		return $ret;
    }    
    
    public function beforeSave($physicalEntity) {
        $now = (new \DateTime())->format(\DateTime::ATOM);
        $physicalEntity->UpdateDate = $now;
        
        if (!isset($physicalEntity->CreationDate)) {
            $physicalEntity->CreationDate = $now;
        }    
    }

	public function loginUser(\stdClass $identity) {
        $identityProvider = $this->getProvider($identity); 
		
        $providerKey = $identityProvider->getProviderKey();
        
        // normalize the identity
        $identityProvider->normalize($identity);        
        
        // look for the physical entity with the provided identity
        $userByIdentity = $this->physicalEntityRepository->findOneByIdentity($providerKey, $identity->$providerKey->LoginKey);
		
        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }
        
        // se esiste, ritorno il documento parziale al chiamante

        $identityProvider->checkCredentials($identity, $userByIdentity);

        // aggiungiamo la rappresentanza verso se stesso, se necessario
        // $this->addUserLegalEntity($userByIdentity);  
        
        $this->physicalEntityRepository->save($userByIdentity);
        
        // crea o aggiorna il db dell'utente
        if ($userByIdentity->UserDBRequested) {
            // $this->createOrRefreshUserDB($userByIdentity);
        }

        return $this->_reduceDocument($userByIdentity);
    }
    
    public function forgottenCredentials(string $providerKey, string $email) {
        // look for the physical entity with the provided email
        $userByEmail = $this->physicalEntityRepository->findOneByEmail($email);
		
        if (!$userByEmail) {
            throw new ValidationException(self::USER_NOT_FOUND, "User with the given email not found");
        } 
        
        $identityProvider = $this->getProviderByKey($providerKey);
        
        $result =  $identityProvider->forgottenCredentials($userByEmail);    
        
        $this->physicalEntityRepository->save($userByEmail);

        return $result;
    }
    
    public function changeCredentials(string $providerKey, string $loginKey, \stdClass $changeCredentials) {
        $identityProvider = $this->getProviderByKey($providerKey);     
        
        // look for the physical entity with the provided identity
        $userByIdentity = $this->physicalEntityRepository->findOneByIdentity($providerKey, $loginKey);
		
        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }

        $identityProvider->changeCredentials($userByIdentity, $changeCredentials);
        
        $this->physicalEntityRepository->save($userByIdentity);

        return $this->_reduceDocument($userByIdentity);        
    }  
    
    private function sendRegistrationEmail($mailer, \stdClass $physicalEntity) {
        $mailer->addAddress($physicalEntity->Contacts->Email->Value, $this->getFullName($physicalEntity));
        
        $translator = new \Symfony\Component\Translation\Translator('it');
        $translator->addLoader('yaml', new \Symfony\Component\Translation\Loader\YamlFileLoader());
        $translator->addResource('yaml', __DIR__."/../Resources/translations/messages.en.yml", "en");
        $translator->addResource('yaml', __DIR__."/../Resources/translations/messages.it.yml", "it");
        
        $mailer->Subject = $translator->trans('registration_subject');
        $mailer->Body    = $translator->trans('registration_confirmation_message').$physicalEntity->Contacts->Email->VerificationCode;   
        $mailer->AltBody = $mailer->Body;
        
        return $mailer->send();
    }
    
    private function getFullName(\stdClass $physicalEntity) {
        return $physicalEntity->FirstName." ".$physicalEntity->LastName;
    }
    
    public function confirmRegistration(string $providerKey, string $loginKey, string $verificationCode) {
        $identityProvider = $this->getProviderByKey($providerKey);     
        
        // look for the physical entity with the provided identity
        $userByIdentity = $this->physicalEntityRepository->findOneByIdentity($providerKey, $loginKey);
		
        if (!$userByIdentity) {
            throw new ValidationException(self::LOGIN_IDENTITY_NOT_FOUND, "Safe identity not found");
        }

        $identityProvider->confirmRegistration($userByIdentity, $verificationCode);
        
        $this->physicalEntityRepository->save($userByIdentity);

        return $this->_reduceDocument($userByIdentity);          
    }    
}
