<?php

namespace Schema31\SafeHubLib\Repository;

use Schema31\CouchDBClient\Domain\ViewRequest;

/**
 * Description of PhysicalEntityRepository
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class PhysicalEntityRepository extends BaseRepository {
    
    public function findOneByFiscalCode(string $fiscalCode) {
        $viewRequest = new ViewRequest("physicalEntities", "byFiscalCode");
        $viewRequest->partialKey($fiscalCode);
        $viewRequest->includeDocs(true);
        
        try {
            $viewResponse = $this->couchDBClient->getView($viewRequest);
        } catch (\Exception $e) {
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "Error retrieving physical entity by fiscal code");
        }
        
        $results = $viewResponse->getDocs();
        
        if (count($results) > 1) {
            throw new UnexpectedException(self::NOT_UNIQUE_RESULT_ERROR, "More than one physical entities found by fiscal code");
        }
        
        return count($results) == 0 ? null : $results[0];        
    }
    
    public function findOneByEmail(string $email, bool $verified = true) {
        $viewRequest = new ViewRequest("physicalEntities", "byEmail");
        $viewRequest->partialKey([$email, $verified]);
        $viewRequest->includeDocs(true);
        
        try {
            $viewResponse = $this->couchDBClient->getView($viewRequest);
        } catch (\Exception $e) {
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "Error retrieving physical entity by email");
        }
        
        $results = $viewResponse->getDocs();
        
        if (count($results) > 1) {
            throw new UnexpectedException(self::NOT_UNIQUE_RESULT_ERROR, "More than one physical entities found by email");
        }
        
        return count($results) == 0 ? null : $results[0];        
    }    

    public function findOneByIdentity(string $providerKey, string $loginKey) {
        $viewRequest = new ViewRequest("users", "byProviderId");
        $viewRequest->partialKey([$providerKey, $loginKey]);
        $viewRequest->includeDocs(true);

        try {
            $viewResponse = $this->couchDBClient->getView($viewRequest);
        } catch (\Exception $e) {
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "Error retrieving physical entity by identity");
        }        
        
        $results = $viewResponse->getDocs();
        
        if (count($results) > 1) {
            throw new UnexpectedException(self::NOT_UNIQUE_RESULT_ERROR, "More than one physical entities found by fiscal code");
        }
        
        return count($results) == 0 ? null : $results[0];
    }

    public function save(\stdClass $physicalEntity) {
        try {
            $this->beforeSave($physicalEntity);
            $this->couchDBClient->storeDoc($physicalEntity);
        } catch (\Exception $e) {
            throw new UnexpectedException(self::COUCHDB_GENERIC_ERROR, "Error saving physical entity");
        } 
    }   

    public function generateId(string $fiscalCode) {        
        return "PE.$fiscalCode.".$this->generateUUID();
    }

}
