<?php

namespace Schema31\PrometheusClient\Storage;

use TweedeGolf\PrometheusClient\Storage\StorageAdapterInterface;

class MysqliAdapter implements StorageAdapterInterface {
    const DEFAULT_TABLE_NAME = 'PROMETEUS_CACHED_VALUES';
    /**
     * @var \mysqli
     */
    protected $mysqli;
    /**
     * @var string
     */
    protected $table;

    /**
     * @param string $dsn utilizzare sintassi tipo mysql://user:passwd@host:port/db_name
     * @param string $table nome della tabella dove memorizzare le statistiche
     */
    public function __construct(
        string $dsn,
        string $table = self::DEFAULT_TABLE_NAME
    ) {
        $values = [];
        \preg_match("/^mysql:\/\/([^:]*):([^@]*)@([^:]*):(\d+)\/([^?.]*)/", $dsn, $values);
        list(, $user, $passwd, $host, $port, $db) = $values;
        $this->mysqli = new \mysqli($host, $user, $passwd, $db, $port);
        $this->table = $table;

        $this->init();
    }

    private function init() {
        $this->mysqli->query("CREATE TABLE IF NOT EXISTS {$this->table}(
                metric_key VARCHAR(256) NOT NULL, 
                labels VARCHAR(2048) NOT NULL,
                value FLOAT NULL,
                CONSTRAINT contacts_pk PRIMARY KEY (metric_key, labels) 
            ) ENGINE=MEMORY");
    }

    /**
     * {@inheritdoc}
     */
    public function getValues($key) {
        $stmt = $this->mysqli->prepare("SELECT value, labels FROM {$this->table} WHERE metric_key = ?");
        $stmt->bind_param('s', $key);
        $stmt->execute();
        $data = $stmt->get_result()->fetch_all(\MYSQLI_ASSOC) ?: [];

        $res = \array_map(function ($row) {
            return [$row['value'], \unserialize($row['labels'])];
        }, $data);

        return $res;
    }

    /**
     * {@inheritdoc}
     */
    public function getValue($key, array $labelValues) {
        $stmt = $this->mysqli->prepare("SELECT value FROM {$this->table} WHERE metric_key = ? and labels = ? LIMIT 1");
        $serializedLabels = \serialize($labelValues);
        $stmt->bind_param('ss', $key, $serializedLabels);
        $stmt->execute();

        $value = $stmt->get_result()->fetch_row() ?: [];
        $res = \reset($value);

        return false === $res ? null : $res;
    }

    /**
     * {@inheritdoc}
     */
    public function setValue($key, $value, array $labelValues) {
        $stmt = $this->mysqli->prepare("INSERT INTO {$this->table}(metric_key, labels, value) VALUES (? , ?, ?) ON DUPLICATE KEY UPDATE value = ?");
        $serializedLabels = \serialize($labelValues);
        $stmt->bind_param('ssdd', $key, $serializedLabels, $value, $value);
        $stmt->execute();

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function incValue($key, $inc, $default, array $labelValues) {
        $stmt = $this->mysqli->prepare("INSERT INTO {$this->table}(metric_key, labels, value) VALUES (? , ?, ?) ON DUPLICATE KEY UPDATE value = value + ?");
        $serializedLabels = \serialize($labelValues);
        $defaultValue = \is_callable($default) ? $default() : $default;
        $initialValue = $defaultValue + $inc;
        $stmt->bind_param('ssdd', $key, $serializedLabels, $initialValue, $inc);
        $stmt->execute();
    }

    /**
     * {@inheritdoc}
     */
    public function hasValue($key, array $labelValues) {
        $stmt = $this->mysqli->prepare("SELECT COUNT(*) FROM {$this->table} WHERE metric_key = ? AND labels = ?");
        $serializedLabels = \serialize($labelValues);
        $stmt->bind_param('ss', $key, $serializedLabels);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_row() ?: [];

        return (\reset($row) ?: 0) > 0;
    }

    /**
     * {@inheritdoc}
     */
    public function getPrefix() {
        return '';
    }
}
