<?php

namespace Schema31\PrometheusClient\Storage;

use TweedeGolf\PrometheusClient\Storage\StorageAdapterInterface;

class PDOAdapter implements StorageAdapterInterface {
    const DEFAULT_TABLE_NAME = 'PROMETEUS_CACHED_VALUES';
    /**
     * @var \PDO
     */
    protected $pdo;

    protected $table;

    public function __construct(\PDO $pdo, string $table = self::DEFAULT_TABLE_NAME) {
        $this->pdo = $pdo;
        $this->table = $table;
        $this->init();
    }

    private function init() {
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS {$this->table}(
                metric_key VARCHAR(256) NOT NULL, 
                labels VARCHAR(2048) NOT NULL,
                value FLOAT NULL,
                CONSTRAINT contacts_pk PRIMARY KEY (metric_key, labels) 
            ) ENGINE=MEMORY");
    }

    /**
     * {@inheritdoc}
     */
    public function getValues($key) {
        $stmt = $this->pdo->prepare("SELECT value, labels FROM {$this->table} WHERE metric_key = :key");
        $stmt->execute([
            'key' => $key,
        ]);
        $data = $stmt->fetchAll(\PDO::FETCH_ASSOC) ?: [];

        $res = \array_map(function ($row) {
            return [$row['value'], \unserialize($row['labels'])];
        }, $data);

        return $res;
    }

    /**
     * {@inheritdoc}
     */
    public function getValue($key, array $labelValues) {
        $stmt = $this->pdo->prepare("SELECT value FROM {$this->table} WHERE metric_key = :key and labels = :labels LIMIT 1");
        $stmt->execute([
            'key' => $key,
            'labels' => \serialize($labelValues),
        ]);

        $value = $stmt->fetch(\PDO::FETCH_ASSOC) ?: [];
        $res = \reset($value);

        return false === $res ? null : $res;
    }

    /**
     * {@inheritdoc}
     */
    public function setValue($key, $value, array $labelValues) {
        $stmt = $this->pdo->prepare("INSERT INTO {$this->table}(metric_key, labels, value) VALUES (:key , :labels, :value) ON DUPLICATE KEY UPDATE value = :value");
        $stmt->execute([
            'key' => $key,
            'labels' => \serialize($labelValues),
            'value' => $value,
        ]);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function incValue($key, $inc, $default, array $labelValues) {
        $stmt = $this->pdo->prepare("INSERT INTO {$this->table}(metric_key, labels, value) VALUES (:key , :labels, :value) ON DUPLICATE KEY UPDATE value = value + :inc");
        $defaultValue = \is_callable($default) ? $default() : $default;
        $initialValue = $defaultValue + $inc;
        $stmt->execute([
            'key' => $key,
            'labels' => \serialize($labelValues),
            'value' => $initialValue,
            'inc' => $inc,
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function hasValue($key, array $labelValues) {
        $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM {$this->table} WHERE metric_key = :key AND labels = :labels");
        $stmt->execute([
            'key' => $key,
            'labels' => \serialize($labelValues),
        ]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC) ?: [];

        return (\reset($row) ?: 0) > 0;
    }

    /**
     * {@inheritdoc}
     */
    public function getPrefix() {
        return '';
    }
}
