<?php

namespace Schema31\PrometheusClient\Storage;

use TweedeGolf\PrometheusClient\Storage\AdapterTrait;
use TweedeGolf\PrometheusClient\Storage\StorageAdapterInterface;

class MysqliAdapter implements StorageAdapterInterface {
    use AdapterTrait;

    const DEFAULT_TABLE_NAME = 'PROMETEUS_CACHED_VALUES';
    /**
     * @var \mysqli
     */
    protected $mysqli;
    /**
     * @var string
     */
    protected $table;

    public function __construct(
        string $host,
        string $user,
        string $passwd,
        string $db,
        string $table = self::DEFAULT_TABLE_NAME
    ) {
        $this->mysqli = new \mysqli($host, $user, $passwd, $db);
        $this->table = $table;

        $this->init();
    }

    private function init() {
        $this->mysqli->query("CREATE TABLE IF NOT EXISTS {$this->table}(
                chiave VARCHAR(256) NOT NULL, 
                labels VARCHAR(2048) NOT NULL,
                value FLOAT NULL,
                CONSTRAINT contacts_pk PRIMARY KEY (chiave, labels) 
            ) ENGINE=MEMORY");
    }

    /**
     * {@inheritdoc}
     * @throws \Exception
     */
    public function getValues($key) {
        $stmt = $this->mysqli->prepare("SELECT value, labels FROM {$this->table} WHERE chiave = ?");
        $stmt->bind_param('s', $key);
        $stmt->execute();
        $data = $stmt->get_result()->fetch_all(\MYSQLI_ASSOC) ?: [];

        $res = \array_map(function ($row) {
            return [$row['value'], \unserialize($row['labels'])];
        }, $data);

        return $res;
    }

    /**
     * {@inheritdoc}
     * @throws \Exception
     */
    public function getValue($key, array $labelValues) {
        $stmt = $this->mysqli->prepare("SELECT value FROM {$this->table} WHERE chiave = ? and labels = ? LIMIT 1");
        $serializedLabels = \serialize($labelValues);
        $stmt->bind_param('ss', $key, $serializedLabels);
        $stmt->execute();

        $value = $stmt->get_result()->fetch_row() ?: [];
        $res = \reset($value);

        return false === $res ? null : $res;
    }

    /**
     * {@inheritdoc}
     * @throws \Exception
     */
    public function setValue($key, $value, array $labelValues) {
        $stmt = $this->mysqli->prepare("INSERT INTO {$this->table}(chiave, labels, value) VALUES (? , ?, ?) ON DUPLICATE KEY UPDATE value = ?");
        $serializedLabels = \serialize($labelValues);
        $stmt->bind_param('ssdd', $key, $serializedLabels, $value, $value);
        $stmt->execute();

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function incValue($key, $inc, $default, array $labelValues) {
        $defaultValue = \is_callable($default) ? $default() : $default;
        $value = $this->getValue($key, $labelValues) ?: $defaultValue;
        $newValue = $value + $inc;

        $this->setValue($key, $newValue, $labelValues);
    }

    /**
     * {@inheritdoc}
     * @throws \Exception
     */
    public function hasValue($key, array $labelValues) {
        $stmt = $this->mysqli->prepare("SELECT COUNT(*) FROM {$this->table} WHERE chiave = ? AND labels = ?");
        $serializedLabels = \serialize($labelValues);
        $stmt->bind_param('ss', $key, $serializedLabels);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_row() ?: [];

        return (\reset($row) ?: 0) > 0;
    }

    /**
     * {@inheritdoc}
     */
    public function getPrefix() {
        return '';
    }
}
