<?php

namespace Schema31\PHPOnCouch\Service;

use Schema31\CouchDBClient\Service\CouchDBAdmin;
use Schema31\CouchDBClient\Exception\CouchDBException;

use PHPOnCouch\CouchAdmin;
use Schema31\PHPOnCouch\Service\CustomCouchClient;
use Schema31\CouchDBClient\Domain\SynchronizationReport;

/**
 * Description of PHPOnCouchAdmin
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class PHPOnCouchAdmin extends PHPOnCouchClient implements CouchDBAdmin {
    private $admin;
      
    public function connect(string $url, string $db, array $options = []): void {        
        $this->couch = new CustomCouchClient($url, $db, $options);
        $this->admin = new CouchAdmin($this->couch);
    }

    public function addAdminUser(string $user): void {
        try {
            $this->admin->addDatabaseAdminUser($user);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }

    public function addMemberUser(string $user): void {
        try {
            $this->admin->addDatabaseMemberUser($user);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }

    public function getAdminUsers(): array {
        try {
            $this->admin->getDatabaseAdminUsers();
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }

    public function getMemberUsers(): array {
        try {
            $this->admin->getDatabaseMemberUsers();
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }

    public function removeAdminUser(string $user): void {
        try {
            $this->admin->removeDatabaseAdminUser($user);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }

    public function removeMemberUser(string $user): void {
        try {
            $this->admin->removeDatabaseMemberUser($user);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }                 
    }
    
    /**
     * @throws CouchDBException
     */
    public function synchronizeDocs(string $folder): SynchronizationReport {
        if (!is_dir($folder)) {
            throw new CouchDBException($folder. " is not a folder");
        }
        
        $synchronizationReport = new SynchronizationReport();
        
        $docsToSave = [];
        $docIds = [];
        
		foreach (glob($folder . DIRECTORY_SEPARATOR . "*.json") as $filename) {
			$docOnDisk = json_decode(file_get_contents($filename));
            
            if (!is_object($docOnDisk)) {
                throw new CouchDBException($filename. " does not contain a valid document");
            }
            
            if (!isset($docOnDisk->_id)) {
                throw new CouchDBException($filename. " contains a document without an id");
            } 
            
            if (!isset($docOnDisk->_rev)) {
                throw new CouchDBException($filename. " contains a document without a rev");
            }             
            
            $docIds[] = $docOnDisk->_id;
            
            try {
                $docOnDB = $this->getDoc($docOnDisk->_id);
                
                list($revOnDisk) = explode("-", $docOnDisk->_rev);
                list($revOnDB) = explode("-", $docOnDB->_rev);
                
                if ($revOnDisk <= $revOnDB) {
                    $synchronizationReport->{$revOnDisk < $revOnDB ? "addMisalignedDoc" : "addAlignedDoc"}($docOnDisk->_id);
                    continue;
                }
                
                $synchronizationReport->addUpdatedDoc($docOnDisk->_id);
                
                $docOnDisk->_rev = $docOnDB->_rev;
            } catch (NotFoundException $e) {
                unset($docOnDisk->_rev);
                $synchronizationReport->addNewDoc($docOnDisk->_id);
            }
            
            $docsToSave[] = $docOnDisk;
		}
        
        if (count($docsToSave) > 0) {
            $this->storeDocs($docsToSave);
        }
        
        return $synchronizationReport;
    }

    /**
     * @throws CouchDBException
     */
    public function synchronizeDesignDocs(string $folder, bool $forceDelete = false): SynchronizationReport {
        $synchronizationReport = $this->synchronizeDocs($folder);
        
        if ($forceDelete) {
            $docIds = $synchronizationReport->getDocIds();

            $designDocsIds = $this->couch->getDesignDocsIds();

            $docsToDelete = [];
            
            foreach (array_diff($designDocsIds, $docIds) as $docIdToDelete) {
                $docsToDelete[] = $this->getDoc($docIdToDelete);
                $synchronizationReport->addDeletedDoc($docIdToDelete);
            }
            
            if (count($docsToDelete) > 0) {
                $this->deleteDocs($docsToDelete);
            }
        }
                
        return $synchronizationReport;        
    } 
}
