<?php

namespace Schema31\PHPOnCouch\Service;

use Schema31\CouchDBClient\Service\CouchDBClient;

use Schema31\CouchDBClient\Domain\ViewRequest;
use Schema31\CouchDBClient\Domain\ViewResponse;
use Schema31\CouchDBClient\Domain\MangoQuery;
use Schema31\CouchDBClient\Domain\MangoResponse;

use Schema31\CouchDBClient\Exception\CouchDBException;
use Schema31\CouchDBClient\Exception\CouchDBNotFoundException;
use Schema31\CouchDBClient\Exception\CouchDBForbiddenException;
use Schema31\CouchDBClient\Exception\CouchDBUnauthorizedException;

use PHPOnCouch\Exceptions\CouchNotFoundException;
use PHPOnCouch\Exceptions\CouchForbiddenException;
use PHPOnCouch\Exceptions\CouchUnauthorizedException;

/**
 * Description of PHPOnCouchClient
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class PHPOnCouchClient extends PHPOnCouch implements CouchDBClient {
    
    static $exceptionsMap = [
        CouchNotFoundException::class => CouchDBNotFoundException::class,
        CouchForbiddenException::class => CouchDBForbiddenException::class,
        CouchUnauthorizedException::class => CouchDBUnauthorizedException::class
    ];
      
    public function connect(string $url, string $db, array $options = []): void {        
        $this->couch = new CustomCouchClient($url, $db, $options);
    }
    
    // section database
    
    /**
     * @return array of string (the uuids)
     * 
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function getUuids(int $count = 1): array {
        try {
            return $this->couch->getUuids($count);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }
    
    // section docs
    
    /**
     * @throws CouchDBException
     * @throws CouchDBNotFoundException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function getDoc(string $id) {
        try {
            return $this->couch->getDoc($id);
        } catch (\Exception $e) {
            throw $this->exceptionMapper($e);
        }        
    }
   
    /**
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function storeDoc($doc): void {
        try {
            $response = $this->couch->storeDoc($doc);
            
            $doc->_id = $response->id;
            $doc->_rev = $response->rev;
        } catch (\Exception $e) {
            throw $this->exceptionMapper($e);
        }       
    }
    
    /**
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function storeDocs(array $docs): void {
        try {
            $responses = $this->couch->storeDocs($docs);
            
            foreach (array_values($docs) as $key => $doc) {
                $doc->_id = $responses[$key]->id;
                $doc->_rev = $responses[$key]->rev;
            }
           
        } catch (\Exception $e) {
            throw $this->exceptionMapper($e);
        }         
    }
    
    /**
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function deleteDoc($doc): void {
        try {
            $this->couch->deleteDoc($doc);
        } catch (\Exception $e) {
            throw $this->exceptionMapper($e);
        }       
    }
    
    /**
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function deleteDocs(array $docs): void {
        try {
            $this->couch->deleteDocs($docs);
        } catch (\Exception $e) {
            throw $this->exceptionMapper($e);
        }       
    }
    
    // section views
    
    /**
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function getView(ViewRequest $viewRequest): ViewResponse {
        try {
            
            $response =  $this->couch->getCustomView($viewRequest);

            return new ViewResponse($response->total_rows, $response->offset, $response->rows);
        } catch (\Exception $e) {
            throw $this->exceptionMapper($e);
        }         
    }
    
    // section mango
    
    /**
     * @throws CouchDBException
     * @throws CouchDBForbiddenException
     * @throws CouchDBUnauthorizedException
     */
    public function find(MangoQuery $query): MangoResponse {
        throw new CouchDBException("Not implemented");
    }  
    
    private function exceptionMapper(\Exception $previousException) {
        $previousExceptionClass = get_class($previousException);
        
        if (isset(self::$exceptionsMap[$previousExceptionClass])) {
            $exceptionClass = self::$exceptionsMap[$previousExceptionClass];
            return new $exceptionClass($previousException->getMessage(), $previousException->getCode());
        } else {
            return new CouchDBException($previousException->getMessage(), $previousException->getCode());
        }
    }
}
