<?php

namespace Schema31\PHPOnCouch\Service;

use Schema31\CouchDBClient\Service\CouchDBClient;
use PHPOnCouch\CouchClient;

use Schema31\CouchDBClient\Domain\ViewRequest;
use Schema31\CouchDBClient\Domain\ViewResponse;
use Schema31\CouchDBClient\Domain\MangoQuery;
use Schema31\CouchDBClient\Domain\MangoResponse;
use Schema31\CouchDBClient\Domain\DocReference;
use Schema31\CouchDBClient\Exception\CouchDBException;

/**
 * Description of PHPOnCouchClient
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class PHPOnCouchClient implements CouchDBClient {
    
    private $client;
    
    public function __construct(string $dsn, string $dbname, array $options = []) {
        if (!isset($options["username"]) && isset($options["password"])) {
            throw new \Exception("Username and password required");
        }
        
        $this->client = new CouchClient($dsn, $dbname, $options);
    }

    
    // section database
    
    /**
     * @return array of string (the uuids)
     * @throws CouchDBException
     */
    public function getUuids(int $count = 1): array {
        try {
            return $this->client->getUuids($count);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }
    }
    
    // section docs
    
    /**
     * @throws CouchDBException
     */
    public function getDoc(string $id) {
        try {
            return $this->client->getDoc($id);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }        
    }
   
    /**
     * @throws CouchDBException
     */
    public function storeDoc($doc): DocReference {
        try {
            $response = $this->client->storeDoc($doc);
            
            return new DocReference($response->id, $response->rev);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }        
    }
    
    /**
     * @return array of DocReference
     * @throws CouchDBException
     */
    public function storeDocs(array $docs): array {
        try {
            $responses = $this->client->storeDocs($docs);
            
            $docReferences = [];
            
            foreach ($responses as $response) {
                $docReferences[] = new DocReference($response->id, $response->rev);
            }
            
            return $docReferences;
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }          
    }
    
    /**
     * @throws CouchDBException
     */
    public function deleteDoc($doc): void {
        try {
            $this->client->deleteDoc($doc);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }        
    }
    
    /**
     * @throws CouchDBException
     */
    public function deleteDocs(array $docs): void {
        try {
            $this->client->deleteDocs($docs);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }         
    }
    
    // section views
    
    /**
     * @throws CouchDBException
     */
    public function getView(ViewRequest $viewRequest): ViewResponse {
        try {
            if ($viewRequest->getStartkey()) {
                $this->client->startkey($viewRequest->getStartkey());
            }

            if ($viewRequest->getEndkey()) {
                $this->client->endkey($viewRequest->getEndkey());
            }

            if ($viewRequest->getSkip()) {
                $this->client->skip($viewRequest->getSkip());
            }

            if ($viewRequest->getLimit()) {
                $this->client->limit($viewRequest->getLimit());
            }

            if ($viewRequest->getDescending()) {
                $this->client->descending($viewRequest->getDescending());
            }   
            
            if ($viewRequest->getIncludeDocs()) {
                $this->client->include_docs($viewRequest->getIncludeDocs());
            }   
            
            $response = $this->client->getView($viewRequest->getDesign(), $viewRequest->getName());
            return new ViewResponse($response->total_rows, $response->offset, $response->rows);
        } catch (\Exception $e) {
            throw new CouchDBException($e->getMessage(), $e->getCode(), $e->getPrevious());
        }         
    }
    
    // section mango
    
    /**
     * @throws CouchDBException
     */
    public function find(MangoQuery $query): MangoResponse {
        return array();
    }

}
