<?php

namespace Schema31\P7MDecoder;

use Schema31\P7MDecoder\Exception\P7MDecodeError;

/**
 * Description of P7MDecoder
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class P7MDecoder implements P7MDecoderInterface {
    
    const CADirectory = __DIR__ . '/../cert/caitalia.pem';
    const FLAGS = PKCS7_NOVERIFY | PKCS7_NOCHAIN | PKCS7_NOSIGS;
    const MIME_TYPES = array("application/pkcs7-mime", "application/octet-stream");
    
    public function decodeContent($p7mContent, $signatureInfo = false, $nestedSignatures = false) {
        $p7mFilepath = tempnam(sys_get_temp_dir(), "tmpP7M");
        
        file_put_contents($p7mFilepath, $p7mContent);
        
        return $this->decodeFile($p7mFilepath, $signatureInfo, $nestedSignatures);
    }

    public function decodeFile($p7mFilepath, $signatureInfo = false, $nestedSignatures = false) {        
        // wrap the file in s/mime
		$sMimeFilePath = $this->derToSMime($p7mFilepath);
        
        // certification file
        $outputCertFile = tempnam(sys_get_temp_dir(), "tmpCert");
        
        // extracted file
        $outputDocumentFile = tempnam(sys_get_temp_dir(), "tmpDoc");

        $resp = openssl_pkcs7_verify($sMimeFilePath, self::FLAGS, $outputCertFile, array(self::CADirectory), self::CADirectory, $outputDocumentFile);
        
		if (file_exists($sMimeFilePath)) {
			unlink($sMimeFilePath);
        }
        
        if ($resp === -1) {
            $resp = $this->openSSLVerify($p7mFilepath, $outputDocumentFile, $outputCertFile);
        }
        
        if ($resp === TRUE && $signatureInfo) {
            $signInfo = $this->parseCertificates($outputCertFile);
        } else {
            $signInfo = array();
        }
        
        if ($resp !== TRUE) {
            if (file_exists($outputDocumentFile)) {
                unlink($outputDocumentFile);
            } 
            
            $outputDocumentFile = NULL;
        }
        
		if (file_exists($outputCertFile)) {
			unlink($outputCertFile);
        }
        
        if ($resp === -1) {
            throw new P7MDecodeError($this->getErrors());
        }
        
        if ($resp && $nestedSignatures) {
            $finfo = new \finfo(FILEINFO_MIME_TYPE);
            $mimeType = $finfo->file($outputDocumentFile);
            
            if (in_array($mimeType, self::MIME_TYPES) || $mimeType === FALSE) {
                try {
                    $innerResult = $this->decodeFile($outputDocumentFile, $signatureInfo, $nestedSignatures);

                    if ($innerResult->isSignatureValid()) {
                        $outputDocumentFile = $innerResult->getDecodedFile();
                        $signInfo = array_merge($signInfo, $innerResult->getSignInfo());
                    }
                } catch (\Exception $e) {
                    // the inner content was not signed (or an error occurred)
                }
            }
        }
          
        return new P7MDecodeResult($resp, $outputDocumentFile, $signInfo);        
    }

    public function verifyContentSignature($p7mContent) {
        $p7mFilepath = tempnam(sys_get_temp_dir(), "tmpP7M");
        
        file_put_contents($p7mFilepath, $p7mContent);
        
        return $this->verifyFileSignature($p7mFilepath);        
    }

    public function verifyFileSignature($p7mFilepath) {
        // wrap the file in s/mime
		$sMimeFilePath = $this->derToSMime($p7mFilepath);
        
        // certification file
        $outputCertFile = tempnam(sys_get_temp_dir(), "tmpCert");

        $resp = openssl_pkcs7_verify($sMimeFilePath, self::FLAGS, $outputCertFile, array(self::CADirectory), self::CADirectory);
        
		if (file_exists($sMimeFilePath)) {
			unlink($sMimeFilePath);
        }
        
		if (file_exists($outputCertFile)) {
			unlink($outputCertFile);
        }
        
        if ($resp === -1) {
            throw new P7MDecodeError($this->getErrors());
        }        
        
        return $resp;        
    }
    
    private function derToSMime($p7mFilepath) {
        $sMimeFilePath = tempnam(sys_get_temp_dir(), "tmpSMime");
		$handle = fopen($sMimeFilePath, "w");
		
        $fileName = basename($p7mFilepath);
		$inputFileContentBase64 = chunk_split(base64_encode(file_get_contents($p7mFilepath)));
		$sMimeMessage = "MIME-Version: 1.0\nContent-Disposition: attachment; filename=\"{$fileName}\"\nContent-Type: application/x-pkcs7-mime; name=\"{$fileName}\"\nContent-Transfer-Encoding: base64\n\n{$inputFileContentBase64}";
        
		fwrite($handle, $sMimeMessage);
		fclose($handle);
        
        return $sMimeFilePath;       
    }
    
    private function getErrors() {
        $errors = "";    
        
        while ($msg = openssl_error_string()) {
            $errors .= "$msg\n";        
        }
        
        return $errors;
    }
    
    private function parseCertificates($certFilePath) {
		$signInfo = array();

		$certs = explode("-----END CERTIFICATE-----\n", file_get_contents($certFilePath));

		foreach ($certs as $cert) {

            if ($cert == '') {
				continue;
            }

			$cert .= '-----END CERTIFICATE-----';
			$cert = openssl_x509_parse($cert);
            
            if (isset($cert["extensions"])) {
                unset($cert["extensions"]);
            }
            
            if (isset($cert["purposes"])) {
                unset($cert["purposes"]);
            }

            if (isset($cert["validFrom_time_t"])) {
                $validFrom = new \DateTime();
                $validFrom->setTimestamp($cert["validFrom_time_t"]);
                $cert["validFrom"] = $validFrom->format(\DateTime::ATOM);
                unset($cert["validFrom_time_t"]);
            }
            
            if (isset($cert["validTo_time_t"])) {
                $validTo = new \DateTime();
                $validTo->setTimestamp($cert["validTo_time_t"]);
                $cert["validTo"] = $validTo->format(\DateTime::ATOM);
                unset($cert["validTo_time_t"]);
            }  
            
            if (isset($cert["issuer"])) {
                $cert["issuer"] = (object) $cert["issuer"];
            }
            
            if (isset($cert["subject"])) {
                $cert["subject"] = (object) $cert["subject"];
            }            
            
            $signInfo[] = (object) $cert;
        }
        
        return $signInfo;
    } 
    
    private function openSSLVerify($p7mFilepath, $outputDocumentFile, $signerCertFile) {
        // $command = "openssl smime -verify -noverify -nosigs -nochain -in $p7mFilepath -inform DER -out $outputDocumentFile -signer $signerCertFile -CAfile $caFile";
        $command = "openssl cms -verify -noverify -in $p7mFilepath -inform DER -out $outputDocumentFile -signer $signerCertFile";
        
        $commandResult = $this->procOpenHandler($command);
        
        $status = $commandResult->status && (strpos($commandResult->stdOut, 'Verification successful') !== false || strpos($commandResult->stdErr, 'Verification successful') !== false);
        
        if ($status !== TRUE) {
            throw new P7MDecodeError($commandResult->stdErr);
        }
        
        return TRUE;      
    }

    /**
     * Background processes handler
     * @param string $command
     * @param string $stdin 
     * @param int $maxExecutionTime
     * @return \stdClass
     */
    private function procOpenHandler($command = '', $stdin = '', $maxExecutionTime = 30) {

        $timeLimit = (time() + $maxExecutionTime);

        $descriptorSpec = array(
            0 => array("pipe", "r"),
            1 => array('pipe', 'w'),
            2 => array('pipe', 'w')
        );

        $pipes = array();

        $response = new \stdClass();
        $response->status = TRUE;
        $response->stdOut = '';
        $response->stdErr = '';
        $response->exitCode = '';

        $process = proc_open($command, $descriptorSpec, $pipes);
        if (!$process) {
            // could not exec command
            $response->status = FALSE;
            return $response;
        }

        $txOff = 0;
        $txLen = strlen($stdin);
        $stdoutDone = FALSE;
        $stderrDone = FALSE;

        // Make stdin/stdout/stderr non-blocking
        stream_set_blocking($pipes[0], 0);
        stream_set_blocking($pipes[1], 0);
        stream_set_blocking($pipes[2], 0);

        if ($txLen == 0) {
            fclose($pipes[0]);
        }

        while (TRUE) {

            if (time() > $timeLimit) {
                // max execution time reached
                // echo 'MAX EXECUTION TIME REACHED'; die;
                @proc_close($process);
                $response->status = FALSE;
                break;
            }

            $rx = array(); // The program's stdout/stderr

            if (!$stdoutDone) {
                $rx[] = $pipes[1];
            }

            if (!$stderrDone) {
                $rx[] = $pipes[2];
            }

            $tx = array(); // The program's stdin

            if ($txOff < $txLen) {
                $tx[] = $pipes[0];
            }

            $ex = NULL;

            stream_select($rx, $tx, $ex, NULL, NULL); // Block til r/w possible

            if (!empty($tx)) {

                $txRet = fwrite($pipes[0], substr($stdin, $txOff, 8192));

                if ($txRet !== FALSE) {
                    $txOff += $txRet;
                }
                if ($txOff >= $txLen) {
                    fclose($pipes[0]);
                }
            }

            foreach ($rx as $r) {

                if ($r == $pipes[1]) {

                    $response->stdOut .= fread($pipes[1], 8192);

                    if (feof($pipes[1])) {

                        fclose($pipes[1]);
                        $stdoutDone = TRUE;
                    }
                } else if ($r == $pipes[2]) {

                    $response->stdErr .= fread($pipes[2], 8192);

                    if (feof($pipes[2])) {

                        fclose($pipes[2]);
                        $stderrDone = TRUE;
                    }
                }
            }
            if (!is_resource($process)) {
                $txOff = $txLen;
                // echo 'PROCESS LOST'; die;
                // break;
            }

            $processStatus = proc_get_status($process);
            if (array_key_exists('running', $processStatus) && !$processStatus['running']) {
                $txOff = $txLen;
                // break;
            }

            if ($txOff >= $txLen && $stdoutDone && $stderrDone) {
                break;
            }
        }

        // Ok - close process (if already running)
        $response->exitCode = @proc_close($process);

        return $response;
    }    
}
