<?php

namespace Schema31\P7MDecoder;

use Schema31\P7MDecoder\Exception\P7MDecodeError;

/**
 * Description of P7MDecoder
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class P7MDecoder implements P7MDecoderInterface {
    
    const CADirectory = __DIR__ . '/../cert/caitalia.pem';
    const FLAGS = PKCS7_NOVERIFY | PKCS7_NOCHAIN | PKCS7_NOSIGS;
    
    public function decodeContent($p7mContent, $signatureInfo = false) {
        $p7mFilepath = tempnam(sys_get_temp_dir(), "tmpP7M");
        
        file_put_contents($p7mFilepath, $p7mContent);
        
        return $this->decodeFile($p7mFilepath, $signatureInfo);
    }

    public function decodeFile($p7mFilepath, $signatureInfo = false) {        
        // wrap the file in s/mime
		$sMimeFilePath = $this->derToSMime($p7mFilepath);
        
        // certification file
        $outputCertFile = tempnam(sys_get_temp_dir(), "tmpCert");
        
        // extracted file
        $outputDocumentFile = tempnam(sys_get_temp_dir(), "tmpDoc");

        $resp = openssl_pkcs7_verify($sMimeFilePath, self::FLAGS, $outputCertFile, array(self::CADirectory), self::CADirectory, $outputDocumentFile);
        
		if (file_exists($sMimeFilePath)) {
			unlink($sMimeFilePath);
        }
        
        if ($resp === TRUE && $signatureInfo) {
            $signInfo = $this->parseCertificates($outputCertFile);
        } else {
            $signInfo = array();
        }
        
        if ($resp !== TRUE) {
            if (file_exists($outputDocumentFile)) {
                unlink($outputDocumentFile);
            } 
            
            $outputDocumentFile = NULL;
        }
        
		if (file_exists($outputCertFile)) {
			unlink($outputCertFile);
        }
        
        if ($resp === -1) {
            throw new P7MDecodeError($this->getErrors());
        }
          
        return new P7MDecodeResult($resp, $outputDocumentFile, $signInfo);        
    }

    public function verifyContentSignature($p7mContent) {
        $p7mFilepath = tempnam(sys_get_temp_dir(), "tmpP7M");
        
        file_put_contents($p7mFilepath, $p7mContent);
        
        return $this->verifyFileSignature($p7mFilepath);        
    }

    public function verifyFileSignature($p7mFilepath) {
        // wrap the file in s/mime
		$sMimeFilePath = $this->derToSMime($p7mFilepath);
        
        // certification file
        $outputCertFile = tempnam(sys_get_temp_dir(), "tmpCert");

        $resp = openssl_pkcs7_verify($sMimeFilePath, self::FLAGS, $outputCertFile, array(self::CADirectory), self::CADirectory);
        
		if (file_exists($sMimeFilePath)) {
			unlink($sMimeFilePath);
        }
        
		if (file_exists($outputCertFile)) {
			unlink($outputCertFile);
        }
        
        if ($resp === -1) {
            throw new P7MDecodeError($this->getErrors());
        }        
        
        return $resp;        
    }
    
    private function derToSMime($p7mFilepath) {
        $sMimeFilePath = tempnam(sys_get_temp_dir(), "tmpSMime");
		$handle = fopen($sMimeFilePath, "w");
		
        $fileName = basename($p7mFilepath);
		$inputFileContentBase64 = chunk_split(base64_encode(file_get_contents($p7mFilepath)));
		$sMimeMessage = "MIME-Version: 1.0\nContent-Disposition: attachment; filename=\"{$fileName}\"\nContent-Type: application/x-pkcs7-mime; name=\"{$fileName}\"\nContent-Transfer-Encoding: base64\n\n{$inputFileContentBase64}";
        
		fwrite($handle, $sMimeMessage);
		fclose($handle);
        
        return $sMimeFilePath;       
    }
    
    private function getErrors() {
        $errors = "";    
        
        while ($msg = openssl_error_string()) {
            $errors .= "$msg\n";        
        }
        
        return $errors;
    }
    
    private function parseCertificates($certFilePath) {
		$signInfo = array();

		$certs = explode("-----END CERTIFICATE-----\n", file_get_contents($certFilePath));

		foreach ($certs as $cert) {

            if ($cert == '') {
				continue;
            }

			$cert .= '-----END CERTIFICATE-----';
			$cert = openssl_x509_parse($cert);
            
            if (isset($cert["extensions"])) {
                unset($cert["extensions"]);
            }
            
            if (isset($cert["purposes"])) {
                unset($cert["purposes"]);
            }

            if (isset($cert["validFrom_time_t"])) {
                $validFrom = new \DateTime();
                $validFrom->setTimestamp($cert["validFrom_time_t"]);
                $cert["validFrom"] = $validFrom->format(\DateTime::ATOM);
                unset($cert["validFrom_time_t"]);
            }
            
            if (isset($cert["validTo_time_t"])) {
                $validTo = new \DateTime();
                $validTo->setTimestamp($cert["validTo_time_t"]);
                $cert["validTo"] = $validTo->format(\DateTime::ATOM);
                unset($cert["validTo_time_t"]);
            }  
            
            if (isset($cert["issuer"])) {
                $cert["issuer"] = (object) $cert["issuer"];
            }
            
            if (isset($cert["subject"])) {
                $cert["subject"] = (object) $cert["subject"];
            }            
            
            $signInfo[] = (object) $cert;
        }
        
        return $signInfo;
    }    

}
