<?php

namespace Schema31\GeoBundle\Type;

use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\FormView;
use Symfony\Component\Validator\Constraints\NotNull;

use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Query\Expr\OrderBy;

use Schema31\GeoBundle\Entity\GeoComune;

use Schema31\GeoBundle\Schema31GeoBundle;

/**
 * Descrizione di LuogoType
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class LuogoType extends AbstractType {

    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder->add('stato', GeoStatoType::class, ['cessati' => $options['cessati'], 'constraints' => [new NotNull()]]);
        
        $formModifier = function(FormInterface $form, ?int $statoId, ?int $provinciaId) use ($options) {      
            $hasProvincia = !is_null($statoId) && $statoId == Schema31GeoBundle::ITALIA_ID;
            
            $constraintsProvinciaComune = $hasProvincia ? [new NotNull()] : [];
                   
            $optionsProvincia = ['cessati' => $options['cessati']];
            $optionsProvincia['disabled'] = !$hasProvincia;
            $optionsProvincia['required'] = true;
            $optionsProvincia['constraints'] = $constraintsProvinciaComune;
                        
            $optionsComune = ['class' => GeoComune::class, 'placeholder' => '','constraints' => $constraintsProvinciaComune];
            if (is_null($provinciaId)) {
                $optionsComune['choices'] = [];
                $optionsComune['disabled'] = true;
                $optionsComune['required'] = true;
            } else {
                $optionsComune['disabled'] = false;
                $optionsComune['required'] = true;
                
                $optionsComune['query_builder'] = function (EntityRepository $er) use ($provinciaId, $options){
                    $orderBy = new OrderBy();
                    $orderBy->add('g.capoluogo', 'DESC');
                    $orderBy->add('g.denominazione', 'ASC');
                    
                    $qb = $er->createQueryBuilder('g')
                    ->where('g.suddivisioneSuperiore = '.$provinciaId);
                    
                    if (!$options['cessati']) {
                        $qb->andWhere('g.cessato = 0');
                    }
                    
                    return $qb->orderBy($orderBy);
                };
            }         
            
            $form->add('provincia', GeoProvinciaType::class, $optionsProvincia);
            $form->add('comune', EntityType::class, $optionsComune);  
            
            $constraintsLocalitaEstera = $hasProvincia ? [] : [new NotNull()];
                          
            $form->add('localitaEstera', null, ['label' => 'Località estera', 'disabled' => $hasProvincia, 'required' => true, 'constraints' => $constraintsLocalitaEstera]); 
        };
           
        $builder->addEventListener(
            FormEvents::PRE_SET_DATA,
            function (FormEvent $event) use ($formModifier){
                $stato = $event->getData()->getStato();
                $provincia = $event->getData()->getProvincia();
                
                $formModifier($event->getForm(), is_null($stato) ? null : $stato->getId(), is_null($provincia) ? null : $provincia->getId());
            }                
        );
        
        $builder->addEventListener(
            FormEvents::PRE_SUBMIT, 
            function (FormEvent $event) use ($formModifier){
                $data = $event->getData();

                $formModifier($event->getForm(), isset($data['stato']) ? (int) $data['stato'] : null, isset($data['provincia']) ? (int) $data['provincia'] : null);
            }
        ); 
        
        $builder->addEventListener(
            FormEvents::POST_SUBMIT, 
            function (FormEvent $event) {
                $data = $event->getData();
               
                if (is_null($data->getStato())) {
                    return;
                }
                
                if ($data->getStato()->getId() === Schema31GeoBundle::ITALIA_ID) {
                    if ($data->getProvincia()) {
                        $data->setRegione($data->getProvincia()->getSuddivisioneSuperiore());
                    }                    
                    $data->setLocalitaEstera(null);
                } else {                   
                    $data->setRegione(null);
                    $data->setProvincia(null);
                    $data->setComune(null);
                }
            }
        );         
    }
    
    public function buildView(FormView $view, FormInterface $form, array $options) {   
        $view->vars['cessati'] = $options['cessati'];
    }
    
    public function configureOptions(OptionsResolver $resolver) 
    {
        $resolver->setDefaults(array(
            'cessati' => false   
        ));       
    } 
    
    public function getParent() {
        return \Symfony\Component\Form\Extension\Core\Type\FormType::class;
    }

}
