<?php

declare(strict_types=1);

namespace Schema31\EnumBundle\Doctrine\DBAL\Type;

use Doctrine\DBAL\DBALException;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\Type;
use function implode;
use InvalidArgumentException;
use function is_string;
use function is_subclass_of;
use MyCLabs\Enum\Enum;
use function sprintf;

class EnumType extends Type {
    /** @var string */
    private $name;
    /** @var string */
    protected $enumClass = Enum::class;

    public function getName(): string {
        return $this->name ?: 'enum';
    }

    /**
     * @param mixed[] $fieldDeclaration
     * @return string
     */
    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform) {
        // return $platform->getVarcharTypeDeclarationSQL($fieldDeclaration);

        $keys = [$this->enumClass, 'keys'];

        $values = implode(', ', array_map(function ($value) {
            return "'$value'";
        }, $keys()));

        return "ENUM($values)";
    }

    /**
     * @param mixed $value the value to convert
     * @return mixed
     * @throws InvalidArgumentException
     */
    public function convertToPHPValue($value, AbstractPlatform $platform) {
        if (null === $value) {
            return null;
        }

        // Check if the value is valid for this enumeration
        /** @var callable $isValidCallable */
        $isValidCallable = [$this->enumClass, 'isValidKey'];
        $isValid = $isValidCallable($value);
        if (!$isValid) {
            $keys = [$this->enumClass, 'keys'];
            throw new InvalidArgumentException(sprintf('The value "%s" is not valid for the enum "%s". Expected one of ["%s"]', $value, $this->enumClass, implode('", "', $keys())));
        }

        return $this->enumClass::$value();
    }

    public function convertToDatabaseValue($value, AbstractPlatform $platform) {
        if (null === $value) {
            return null;
        }

        // Otherwise, cast to string
        return (string) $value->getKey();
    }

    /**
     * @throws InvalidArgumentException
     * @throws DBALException
     */
    public static function registerEnumType(string $typeNameOrEnumClass, ?string $enumClass = null): void {
        $typeName = $typeNameOrEnumClass;
        $enumClass = $enumClass ?: $typeNameOrEnumClass;

        if (!is_subclass_of($enumClass, Enum::class)) {
            throw new InvalidArgumentException(sprintf('Provided enum class "%s" is not valid. Enums must extend "%s"', $enumClass, Enum::class));
        }

        // Register and customize the type
        self::addType($typeName, static::class);
        /** @var EnumType $type */
        $type = self::getType($typeName);
        $type->name = $typeName;
        $type->enumClass = $enumClass;
    }

    /**
     * @param array<string|int, string> $types
     * @throws \Exception
     * @throws DBALException
     */
    public static function registerEnumTypes(array $types): void {
        foreach ($types as $typeName => $enumClass) {
            $typeName = is_string($typeName) ? $typeName : $enumClass;
            if (!static::hasType($typeName)) {
                static::registerEnumType($typeName, $enumClass);
            }
        }
    }

    public function requiresSQLCommentHint(AbstractPlatform $platform): bool {
        return true;
    }
}
