<?php

namespace Schema31\EnumBundle\Form\DataTransformer;

use Exception;
use MyCLabs\Enum\Enum;
use Symfony\Component\Form\DataTransformerInterface;
use Symfony\Component\Form\Exception\InvalidArgumentException;
use Symfony\Component\Form\Exception\TransformationFailedException;

/**
 * Enumerated values to enum instances transformer.
 */
class ValueToEnumTransformer implements DataTransformerInterface {
    /** @var string */
    private $enumClass;

    public function __construct(string $enumClass) {
        if (!is_a($enumClass, Enum::class, true)) {
            throw new InvalidArgumentException(sprintf('"%s" is not an instance of "%s"', $enumClass, Enum::class));
        }

        $this->enumClass = $enumClass;
    }

    /**
     * Transforms Enum object to a raw enumerated value.
     *
     * @param mixed $value EnumInterface instance
     *
     * @throws TransformationFailedException When the transformation fails
     *
     * @return string|null Value of EnumInterface
     */
    public function transform($value) {
        if (null === $value) {
            return null;
        }

        if (!$value instanceof $this->enumClass) {
            throw new TransformationFailedException(sprintf('Expected instance of "%s". Got "%s".', $this->enumClass, \is_object($value) ? \get_class($value) : \gettype($value)));
        }

        return $value->getKey();
    }

    /**
     * Transforms a raw enumerated value to an enumeration instance.
     *
     * @param mixed $value Value accepted by Enum
     *
     * @throws TransformationFailedException When the transformation fails
     *
     * @return Enum|null A single Enum instance or null
     */
    public function reverseTransform($value) {
        if (null === $value) {
            return null;
        }

        try {
            return $this->enumClass::$value();
        } catch (Exception $exception) {
            throw new TransformationFailedException($exception->getMessage(), $exception->getCode(), $exception);
        }
    }
}
