<?php

namespace Schema31\CouchDBJobManager\Service;

use Schema31\JobManager\Service\JobManager;
use Schema31\CouchDBJobManager\Domain\CouchDBJob;

/**
 * Description of CouchDBJobManager
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class CouchDBJobManager implements JobManager {
    
    private $jobsToSave;
    private $client;
    
    public function __construct(string $url, string $dbname, string $user, string $password) {
        $this->client = \Doctrine\CouchDB\MangoClient::create(array('url' => $url, 'dbname' => $dbname, 'user' => $user, 'password' => $password));
        $this->mapper = new CouchDBMapper();
        $this->jobToSave = array();
    }
    
    public function createJob(string $topic, string $submitter, string $status, array $details) {
        return new CouchDBJob($topic, $submitter, $status, $details);
    }

    public function addJob($job) {
        if (!is_null($job->getId())) {
            throw new \Exception("Job was already saved");
        }
  
        $uuids = $this->client->getUuids();
        $uuid = $uuids[0]; 
        $job->setId($uuid);
        $this->jobsToSave[$uuid] = $job;
    }

    public function modifyJob($job) {
        if (is_null($job->getId())) {
            throw new \Exception("Job was not yet saved");
        }  
        
        $this->jobsToSave[$job->getId()] = $job;
    }

    public function saveJobs() {
        $bulkUpdater = $this->client->createBulkUpdater();
                
        $documents = array();
        foreach ($this->jobsToSave as $job) {
            $documents[] = $this->mapper->map($job);
        }
        
        $bulkUpdater->updateDocuments($documents);
        $response = $bulkUpdater->execute();

        if ($response->status >= 200 && $response->status < 300 && isset($response->body)) {
            foreach ($response->body as $doc) {
                $this->jobsToSave[$doc["id"]]->setRev($doc["rev"]);
            }
            
            $this->jobToSave = array();
            
            return TRUE;
        } else {
            throw new \Exception("CouchDB query error");
        }        
    }
    
    public function findJobsByTopicAndStatus(string $topic, string $status) {
        $selector = array('topic'=> $topic, 'status' => $status);
        $index = null; // array('_design/jobs', 'byTopicAndStatus');
        return $this->findJobsBySelector($selector, $index);  
    }    

    public function findJobsByTopicAndSubmitter(string $topic, string $submitter) {
        $selector = array('topic'=> $topic, 'submitter' => $submitter);
        return $this->findJobsBySelector($selector);       
    }
    
    private function findJobsBySelector(array $selector, array $index = null) {
        $options = array();
        
        if (!is_null($index)) {
            $options['use_index'] = $index;
        }
        
        $query = new \Doctrine\CouchDB\Mango\MangoQuery($selector, $options);
        $response = $this->client->find($query);    
        
        if ($response->status === 200 && isset($response->body["docs"])) {
            
            $jobs = array();
            foreach ($response->body["docs"] as $doc) {
                $jobs[] = $this->mapper->inverseMap($doc, "Schema31\CouchDBJobManager\Domain\CouchDBJob");
            }
            
            return $jobs;
        } else {
            throw new \Exception("CouchDB query error");
        }        
    }

}
