<?php

namespace Schema31\CouchDBClient\Domain;

use Schema31\Exceptions\Operation\UnexpectedException;

/**
 * Description of ViewRequest
 *
 * @author Antonio Turdo <aturdo@schema31.it>
 */
class ViewRequest {
    
    private $design;
    
    private $name;
    
    private $startkey;
    
    private $endkey;
    
    private $skip;
    
    private $limit;
    
    private $reduce;
    
    private $includeDocs;
    
    private $descending;
    
    private $key;
    
    private $groupLevel;
    
    public function __construct(?string $design = null, ?string $name = null) {
        if (($design && !$name) || (!$design && $name)) {
            throw new UnexpectedException("design and name must be provided both or none");
        }
        
        $this->design = $design;
        $this->name = $name;
        $this->reduce = false;
    }
    
    /**
     * @param string|array $key
     * @return ViewRequest
     * @throws \Exception
     */
    public function partialKey($key) {
        $this->startkey($key);
        $this->endkey($key);
        
        return $this;
    }
    
    /**
     * @param string|array $key
     * @return ViewRequest
     * @throws \Exception
     */
    public function key($key) {
        if (is_array($key)) {
            $key = json_encode($key);
        } elseif (is_string($key)) {
            $key = '"'.$key.'"';
        }
        
        $this->key = $key;
        
        return $this;        
    }
    
    /**
     * @param string|array $startkey
     * @throws \Exception
     */
    public function startkey($startkey): ViewRequest {
        if (!is_string($startkey) && !is_array($startkey)) {
            throw new \Exception("Invalid argument type");
        }
        
        if (is_array($startkey)) {
            $startkey = json_encode($startkey);
        } elseif (is_string($startkey)) {
            $startkey = '"'.$startkey.'"';
        }
        
        $this->startkey = $startkey;
        return $this;
    }
   
    /**
     * @param string|array $endkey
     * @return ViewRequest
     * @throws \Exception
     */    
    public function endkey($endkey): ViewRequest {
        if (!is_string($endkey) && !is_array($endkey)) {
            throw new \Exception("Invalid argument type");
        }
        
        if (is_array($endkey)) {
            $endkey = substr(json_encode($endkey), 0, -1).',{}]';
        } elseif (is_string($endkey)) {
            $endkey = '"'.$endkey.'\ufff0"';
        }       
        
        $this->endkey = $endkey;
        return $this;
    }

    public function skip(int $skip): ViewRequest {
        $this->skip = $skip;
        return $this;
    }

    public function limit(int $limit): ViewRequest {
        $this->limit = $limit;
        return $this;
    }
    
    public function reduce(bool $reduce): ViewRequest {
        $this->reduce = $reduce;
        return $this;
    }

    public function includeDocs(bool $includeDocs): ViewRequest {
        $this->includeDocs = $includeDocs;
        return $this;
    }
    
    public function descending(bool $descending): ViewRequest {
        $this->descending = $descending;
        return $this;
    }
    
    public function groupLevel(int $groupLevel): ViewRequest {
        $this->groupLevel = $groupLevel;
        return $this;
    }    
    
    public function getDesign(): ?string {
        return $this->design;
    }

    public function getName(): ?string {
        return $this->name;
    }

    public function getKey(): ?string {
        return $this->key;
    }

    public function getStartkey(): ?string {
        return $this->startkey;
    }

    public function getEndkey(): ?string {
        return $this->endkey;
    }

    public function getSkip(): ?int {
        return $this->skip;
    }

    public function getLimit(): ?int {
        return $this->limit;
    }
    
    public function getReduce(): bool {
        return $this->reduce;
    }

    public function getIncludeDocs(): ?bool {
        return $this->includeDocs;
    }

    public function getDescending(): ?bool {
        return $this->descending;
    }
    
    public function getGroupLevel(): ?int {
        return $this->groupLevel;
    }    

}
